import { jsPDF } from 'jspdf';
import html2canvas from 'html2canvas';
import { BeamConfig, AnalysisResult } from '../types';

// ─── Drawing Helpers ───────────────────────────────────────────────────────────

interface TableColumn {
    header: string;
    width: number;         // column width in mm
    align?: 'left' | 'right' | 'center';
}

const MARGIN = 15;
const PAGE_WIDTH = 210;     // A4 width mm
const PAGE_HEIGHT = 297;    // A4 height mm
const CONTENT_WIDTH = PAGE_WIDTH - MARGIN * 2;
const ROW_H = 8;             // row height mm
const HEADER_BG = [15, 23, 42] as const;   // slate-900
const ACCENT_COLOR = [37, 99, 235] as const; // blue-600
const ALT_BG = [248, 250, 252] as const;   // slate-50
const TEXT_DARK = [15, 23, 42] as const;   // slate-900
const TEXT_MID = [71, 85, 105] as const;   // slate-500
const TEXT_LIGHT = [148, 163, 184] as const; // slate-400
// Use a robust path for the logo image
const LOGO_PATH = `${window.location.origin}/assets/logo.jpg`;

async function getBase64Image(url: string): Promise<string> {
    const response = await fetch(url);
    const blob = await response.blob();
    return new Promise((resolve, reject) => {
        const reader = new FileReader();
        reader.onloadend = () => resolve(reader.result as string);
        reader.onerror = reject;
        reader.readAsDataURL(blob);
    });
}

function drawWatermark(pdf: jsPDF, logoBase64: string) {
    pdf.saveGraphicsState();
    // Increased opacity for better visibility
    (pdf as any).setGState(new (pdf as any).GState({ opacity: 0.25 }));
    const size = 100;
    const x = (PAGE_WIDTH - size) / 2;
    const y = (PAGE_HEIGHT - size) / 2;
    pdf.addImage(logoBase64, 'JPEG', x, y, size, size, undefined, 'FAST');
    pdf.restoreGraphicsState();
}

function drawSectionHeader(pdf: jsPDF, label: string, title: string, y: number): number {
    pdf.setFont('helvetica', 'bold');
    pdf.setFontSize(12);
    pdf.setTextColor(...ACCENT_COLOR);
    pdf.text(label, MARGIN, y);

    pdf.setTextColor(...TEXT_DARK);
    pdf.text(title, MARGIN + 8, y);

    // Underline
    pdf.setDrawColor(...TEXT_LIGHT);
    pdf.setLineWidth(0.1);
    pdf.line(MARGIN, y + 2, MARGIN + CONTENT_WIDTH, y + 2);

    return y + 10;
}

function drawTableSection(
    pdf: jsPDF,
    title: string,
    columns: TableColumn[],
    rows: (string | number)[][],
    startY: number,
    sectionLabel: string
): number {
    let y = drawSectionHeader(pdf, sectionLabel, title, startY);

    // Table header bar
    pdf.setFillColor(...HEADER_BG);
    pdf.rect(MARGIN, y, CONTENT_WIDTH, ROW_H, 'F');

    pdf.setFont('helvetica', 'bold');
    pdf.setFontSize(8.5);
    pdf.setTextColor(255, 255, 255);

    let colX = MARGIN;
    for (const col of columns) {
        const textX = col.align === 'right'
            ? colX + col.width - 2
            : col.align === 'center'
                ? colX + col.width / 2
                : colX + 2;
        pdf.text(col.header, textX, y + ROW_H - 2, { align: col.align ?? 'left' });
        colX += col.width;
    }
    y += ROW_H;

    // Data rows
    pdf.setFont('helvetica', 'normal');
    pdf.setFontSize(8.5);
    pdf.setTextColor(...TEXT_DARK);

    rows.forEach((row, ri) => {
        // Multi-page check inside table if rows are many
        if (y > 270) {
            pdf.addPage();
            y = 20;
            // Redraw header for the continued table
            pdf.setFillColor(...HEADER_BG);
            pdf.rect(MARGIN, y, CONTENT_WIDTH, ROW_H, 'F');
            pdf.setFont('helvetica', 'bold');
            pdf.setFontSize(8.5);
            pdf.setTextColor(255, 255, 255);
            let cx = MARGIN;
            columns.forEach(col => {
                const tx = col.align === 'right' ? cx + col.width - 2 : col.align === 'center' ? cx + col.width / 2 : cx + 2;
                pdf.text(col.header, tx, y + ROW_H - 2, { align: col.align ?? 'left' });
                cx += col.width;
            });
            y += ROW_H;
            pdf.setFont('helvetica', 'normal');
            pdf.setTextColor(...TEXT_DARK);
        }

        if (ri % 2 === 0) {
            pdf.setFillColor(...ALT_BG);
            pdf.rect(MARGIN, y, CONTENT_WIDTH, ROW_H, 'F');
        }

        colX = MARGIN;
        row.forEach((cell, ci) => {
            const col = columns[ci];
            const cellStr = String(cell);
            const textX = col.align === 'right'
                ? colX + col.width - 2
                : col.align === 'center'
                    ? colX + col.width / 2
                    : colX + 2;
            pdf.text(cellStr, textX, y + ROW_H - 2.5, { align: col.align ?? 'left' });

            // Vertical Divider
            if (ci > 0) {
                pdf.setDrawColor(226, 232, 240); // slate-200
                pdf.setLineWidth(0.05);
                pdf.line(colX, y, colX, y + ROW_H);
            }

            colX += col.width;
        });

        // Row bottom border (very subtle)
        pdf.setDrawColor(241, 245, 249);
        pdf.setLineWidth(0.1);
        pdf.line(MARGIN, y + ROW_H, MARGIN + CONTENT_WIDTH, y + ROW_H);
        y += ROW_H;
    });

    return y + 12;
}

function drawFooter(pdf: jsPDF) {
    const totalPages = (pdf as any).internal.getNumberOfPages();
    const currentPage = (pdf as any).internal.getCurrentPageInfo().pageNumber;

    pdf.setFont('helvetica', 'normal');
    pdf.setFontSize(8);
    pdf.setTextColor(...TEXT_LIGHT);

    const footerY = PAGE_HEIGHT - 10;
    pdf.text(`Professional Structural Analysis Report | CivilBeam Pro`, MARGIN, footerY);
    pdf.text(`Page ${currentPage}`, PAGE_WIDTH - MARGIN, footerY, { align: 'right' });

    // A subtle line above footer
    pdf.setDrawColor(241, 245, 249);
    pdf.setLineWidth(0.1);
    pdf.line(MARGIN, footerY - 3, PAGE_WIDTH - MARGIN, footerY - 3);
}

function checkPageBreak(pdf: jsPDF, y: number, needed: number): number {
    if (y + needed > 275) {
        pdf.addPage();
        return 20;
    }
    return y;
}

// ─── Main Export Function ──────────────────────────────────────────────────────

export const generatePDFReport = async (
    config: BeamConfig,
    results: AnalysisResult,
    visualizerId: string,
    chartsId: string
): Promise<boolean> => {
    try {
        const logoBase64 = await getBase64Image(LOGO_PATH).catch(() => null);
        const pdf = new jsPDF('p', 'mm', 'a4');

        // ── COVER HEADER ──────────────────────────────────────────────────────
        pdf.setFillColor(...HEADER_BG);
        pdf.rect(0, 0, PAGE_WIDTH, 45, 'F');

        // Logo Image in Header
        if (logoBase64) {
            pdf.saveGraphicsState();
            // Circular clip for header logo? No, just square or let it be.
            pdf.addImage(logoBase64, 'JPEG', MARGIN, 5, 24, 24);
            pdf.restoreGraphicsState();
        }

        // Logo / Title
        const titleX = logoBase64 ? MARGIN + 28 : MARGIN;
        pdf.setFont('helvetica', 'bold');
        pdf.setFontSize(24);
        pdf.setTextColor(255, 255, 255);
        pdf.text('CivilBeam', titleX, 18);
        pdf.setTextColor(...ACCENT_COLOR);
        pdf.text('Pro', titleX + 42, 18);

        pdf.setFont('helvetica', 'normal');
        pdf.setFontSize(10);
        pdf.setTextColor(255, 255, 255);
        pdf.text('PRECISE STRUCTRUAL ANALYSIS REPORT', titleX, 26);

        pdf.setDrawColor(...ACCENT_COLOR);
        pdf.setLineWidth(1);
        pdf.line(titleX, 30, titleX + 20, 30);

        pdf.setFontSize(9);
        pdf.setTextColor(200, 200, 200);
        pdf.text(`REPORT ID: CB-${Date.now().toString().slice(-8)}`, PAGE_WIDTH - MARGIN, 15, { align: 'right' });
        pdf.text(`GENERATED: ${new Date().toLocaleString().toUpperCase()}`, PAGE_WIDTH - MARGIN, 21, { align: 'right' });

        // Project Info Bar
        pdf.setFillColor(30, 41, 59); // slate-800
        pdf.rect(0, 45, PAGE_WIDTH, 15, 'F');
        pdf.setTextColor(255, 255, 255);
        pdf.setFontSize(9);
        pdf.text('PROJECT:', MARGIN, 54);
        pdf.setFont('helvetica', 'bold');
        pdf.text('Analysis of Continuous Beam Structure', MARGIN + 18, 54);

        pdf.setFont('helvetica', 'normal');
        pdf.text('STATUS:', PAGE_WIDTH - 60, 54);
        pdf.setTextColor(132, 204, 22); // lime-500
        pdf.text('COMPLETED', PAGE_WIDTH - 45, 54);

        let y = 75;

        // ── SECTION 1: BEAM PROPERTIES ────────────────────────────────────────
        y = drawTableSection(pdf, 'Beam Properties', [
            { header: 'Parameter', width: 85 },
            { header: 'Value', width: 55, align: 'right' },
            { header: 'Unit', width: 40 },
        ], [
            ['Beam Length', config.length, 'm'],
            ['Youngs Modulus (E)', (config.E / 1e6).toFixed(1), 'GPa'],
            ['Moment of Inertia (I)', (config.I * 1e12).toExponential(3), 'mm4'],
            ['Section Depth (d)', config.depth !== undefined ? config.depth : 'N/A', config.depth !== undefined ? 'm' : ''],
        ], y, '1.');

        // ── SECTION 2: SUPPORTS ──────────────────────────────────────────────
        y = checkPageBreak(pdf, y, (config.supports.length + 3) * ROW_H);
        y = drawTableSection(pdf, 'Supports', [
            { header: 'ID', width: 15 },
            { header: 'Position x', width: 35, align: 'right' },
            { header: 'Type', width: 35 },
            { header: 'Settlement (dy)', width: 55, align: 'right' },
            { header: 'Stiffness (k)', width: 40, align: 'right' },
        ], config.supports.map(s => [
            s.id,
            `${s.x} m`,
            s.type,
            s.settlement !== undefined && s.settlement !== 0 ? `${s.settlement} m` : '-',
            s.type === 'spring_y' && s.stiffness !== undefined ? `${s.stiffness} kN/m` : '-',
        ]), y, '2.');

        // ── SECTION 3: LOADS ─────────────────────────────────────────────────
        y = checkPageBreak(pdf, y, (config.loads.length + 3) * ROW_H);
        y = drawTableSection(pdf, 'Applied Loads', [
            { header: 'ID', width: 15 },
            { header: 'Type', width: 20 },
            { header: 'Position / Range', width: 60 },
            { header: 'Magnitude', width: 50, align: 'right' },
            { header: 'Unit', width: 35 },
        ], config.loads.map(l => {
            if (l.type === 'point') return [l.id, 'Point', `x = ${l.x} m`, l.value?.toFixed(2) ?? '0', 'kN'];
            if (l.type === 'moment') return [l.id, 'Moment', `x = ${l.x} m`, l.value?.toFixed(2) ?? '0', 'kNm'];
            if (l.type === 'udl') return [l.id, 'UDL', `${l.start} m to ${l.end} m`, l.value?.toFixed(2) ?? '0', 'kN/m'];
            return [l.id, 'UVL', `${l.start} m to ${l.end} m`, `${l.valueStart?.toFixed(2)} to ${l.valueEnd?.toFixed(2)}`, 'kN/m'];
        }), y, '3.');

        // ── SECTION 4: ANALYSIS SUMMARY ──────────────────────────────────────
        y = checkPageBreak(pdf, y, 70); // Summary + reactions often go together
        const shear = results.diagrams.shear;
        const moment = results.diagrams.moment;
        const defl = results.diagrams.deflection;
        const stress = results.diagrams.bendingStress ?? [];
        const xArr = results.diagrams.x;

        const maxShearIdx = shear.reduce((mi, v, i) => Math.abs(v) > Math.abs(shear[mi]) ? i : mi, 0);
        const maxMomIdx = moment.reduce((mi, v, i) => Math.abs(v) > Math.abs(moment[mi]) ? i : mi, 0);
        const maxDeflIdx = defl.reduce((mi, v, i) => Math.abs(v) > Math.abs(defl[mi]) ? i : mi, 0);
        const maxStressIdx = stress.length > 0 ? stress.reduce((mi, v, i) => Math.abs(v) > Math.abs(stress[mi]) ? i : mi, 0) : -1;

        y = drawTableSection(pdf, 'Analysis Summary', [
            { header: 'Result', width: 70 },
            { header: 'Max (+)', width: 45, align: 'right' },
            { header: 'Min (-)', width: 45, align: 'right' },
            { header: 'Critical x', width: 20, align: 'right' },
        ], [
            ['Shear Force (kN)', Math.max(...shear).toFixed(3), Math.min(...shear).toFixed(3), `${xArr[maxShearIdx]?.toFixed(2)} m`],
            ['Bending Moment (kNm)', Math.max(...moment).toFixed(3), Math.min(...moment).toFixed(3), `${xArr[maxMomIdx]?.toFixed(2)} m`],
            ['Deflection (mm)', (Math.max(...defl) * 1000).toFixed(3), (Math.min(...defl) * 1000).toFixed(3), `${xArr[maxDeflIdx]?.toFixed(2)} m`],
            ...(maxStressIdx >= 0 ? [['Bending Stress (MPa)', (Math.max(...stress) / 1000).toFixed(3), (Math.min(...stress) / 1000).toFixed(3), `${xArr[maxStressIdx]?.toFixed(2)} m`]] : []),
        ], y, '4.');

        // ── SECTION 5: SUPPORT REACTIONS ─────────────────────────────────────
        const reactions = results.reactions ?? {};
        const reactionRows = Object.keys(reactions).map(k => {
            const r = reactions[parseFloat(k)];
            return [`x = ${k} m`, r.Fy.toFixed(3), r.Mz.toFixed(3)];
        });

        if (reactionRows.length > 0) {
            y = checkPageBreak(pdf, y, (reactionRows.length + 3) * ROW_H);
            y = drawTableSection(pdf, 'Support Reactions', [
                { header: 'Support Position', width: 70 },
                { header: 'Vertical Reaction Fy (kN)', width: 65, align: 'right' },
                { header: 'Moment Mz (kNm)', width: 45, align: 'right' },
            ], reactionRows, y, '5.');
        }

        // ── SECTION 6: GRAPHICAL RESULTS ─────────────────────────────────────
        y = checkPageBreak(pdf, y, 120); // Try to fit visualization on same page if possible
        y = drawSectionHeader(pdf, '6.', 'Structural Visualization', y);

        // Beam visualizer
        const visualizerEl = document.getElementById(visualizerId);
        if (visualizerEl) {
            const canvas = await html2canvas(visualizerEl, {
                scale: 4,
                useCORS: true,
                ignoreElements: (el) => el.classList.contains('z-20') || el.tagName === 'BUTTON'
            });
            const imgData = canvas.toDataURL('image/png', 1.0);
            const imgW = CONTENT_WIDTH;
            const imgH = (canvas.height * imgW) / canvas.width;

            // Background for the visualizer to make it look clinical
            pdf.setFillColor(248, 250, 252);
            pdf.rect(MARGIN, y, imgW, imgH, 'F');

            pdf.addImage(imgData, 'PNG', MARGIN, y, imgW, imgH);

            pdf.setFont('helvetica', 'italic');
            pdf.setFontSize(8);
            pdf.setTextColor(...TEXT_MID);
            pdf.text('Figure 1: Beam Configuration and Loading Diagram', PAGE_WIDTH / 2, y + imgH + 5, { align: 'center' });

            y += imgH + 15;
        }

        // Charts
        y = checkPageBreak(pdf, y, 150);
        y = drawSectionHeader(pdf, '7.', 'Analysis Diagrams', y);

        const chartsEl = document.getElementById(chartsId);
        if (chartsEl) {
            const canvas = await html2canvas(chartsEl, {
                scale: 4,
                useCORS: true,
                ignoreElements: (el) => el.classList.contains('apexcharts-toolbar') || el.tagName === 'BUTTON'
            });
            const imgData = canvas.toDataURL('image/png', 1.0);
            const imgW = CONTENT_WIDTH;
            const imgH = Math.min((canvas.height * imgW) / canvas.width, 210);

            y = checkPageBreak(pdf, y, imgH + 10);

            pdf.addImage(imgData, 'PNG', MARGIN, y, imgW, imgH);

            pdf.setFont('helvetica', 'italic');
            pdf.setFontSize(8);
            pdf.setTextColor(...TEXT_MID);
            pdf.text('Figure 2: Shear Force, Bending Moment, Deflection and Stress Diagrams', PAGE_WIDTH / 2, y + imgH + 5, { align: 'center' });
        }

        // ── FINAL PASS: DRAW WATERMARK AND FOOTER ON ALL PAGES ────────────────
        const totalPages = (pdf as any).internal.getNumberOfPages();
        for (let i = 1; i <= totalPages; i++) {
            pdf.setPage(i);
            if (logoBase64) drawWatermark(pdf, logoBase64);
            drawFooter(pdf);
        }

        pdf.save(`CivilBeam_Analysis_Report_CBD-${new Date().getTime()}.pdf`);
        return true;

    } catch (error) {
        console.error('Failed to generate PDF Report', error);
        return false;
    }
};
