import { analyzeBeam, TOLERANCE } from './fem-engine-pro';
import { BeamConfig, ILDResult, ILDTargetType } from '../types';

export interface ILDTarget
 {
    type: ILDTargetType;
    location: number;
    component?: 'Fy' | 'Mz';
}

export function calculateInfluenceLine(originalConfig: BeamConfig, target: ILDTarget, samples = 300, explicitPoints: number[] = []): ILDResult {
  const criticalPoints = new Set<number>();
  criticalPoints.add(0);
  criticalPoints.add(originalConfig.length);
  criticalPoints.add(target.location);
  
  if (explicitPoints) explicitPoints.forEach(p => criticalPoints.add(p));

  const configCopy: any = JSON.parse(JSON.stringify(originalConfig));
  
  // Create spans wrapper if missing for consistency
  if(!configCopy.spans) configCopy.spans = [{length: configCopy.length}];
  
  configCopy.spans.forEach((span: any) => {
    span.supports?.forEach((s: any) => criticalPoints.add(s.x));
    span.hinges?.forEach((h: any) => criticalPoints.add(h.x));
  });

  if (configCopy.sections) {
    configCopy.sections.forEach((sec: any) => {
        criticalPoints.add(sec.x);
        if(sec.end !== undefined) criticalPoints.add(sec.end);
    });
  }

  const L = originalConfig.length;
  // Use a fixed grid step of 0.01m (1cm) to ensure smooth charts and exact value picking
  // Using integer loop to prevent floating point drift which can cause missed end-points (e.g. 11m)
  const step = 0.01;
  const numSteps = Math.ceil(L / step);
  for(let i=0; i<=numSteps; i++) {
      const val = i * step;
      // Round to fix float noise like 5.000000001
      const x = Math.round(val * 100) / 100;
      if (x <= L + TOLERANCE) {
          criticalPoints.add(x);
      }
  }

  const sortedX = Array.from(criticalPoints).sort((a, b) => a - b);
  const uniqueX = [sortedX[0]];
  
  for (let i = 1; i < sortedX.length; i++) {
    const last = uniqueX[uniqueX.length - 1];
    const curr = sortedX[i];
    if (curr - last > 1e-6) { 
      uniqueX.push(curr);
    }
  }

  const xData: number[] = [];
  const yData: number[] = [];

  if (target.type === 'reaction') {
    let supportExists = false;
    configCopy.supports?.forEach((s: any) => {
        if (Math.abs(s.x - target.location) < TOLERANCE) supportExists = true;
    });
    // Also check spans structure if present
    if(!supportExists) {
        configCopy.spans.forEach((s:any) => s.supports?.forEach((sup:any) => {
             if (Math.abs(sup.x - target.location) < TOLERANCE) supportExists = true;
        }));
    }
    
    if (!supportExists) {
        // Just return zeros or handle error gracefully
        console.warn("No support at location for Reaction ILD");
        return { x: uniqueX, value: new Array(uniqueX.length).fill(0) };
    }
  }

  // Pre-clean for Simulation
  // We strictly ONLY remove loads. 
  // We KEEP settlements/prescribed displacements if they exist, as they affect the baseline of the structure in indeterminate cases.
  const simConfig = JSON.parse(JSON.stringify(originalConfig));
  
  // Ensure flat structure logic and clean any nested loads
  if(!simConfig.spans) {
       simConfig.spans = [{
          id: 'main-span',
          length: simConfig.length,
          supports: simConfig.supports || [],
          loads: [],
          hinges: simConfig.hinges || []
      }];
  } else {
      simConfig.spans.forEach((span: any) => {
          if (span.loads) span.loads = [];
          // Note: We do NOT remove settlements/rotations from supports here anymore.
      });
  }
  
  // Clear loads
  simConfig.loads = [];
  simConfig.globalLoads = []; 

  for (const xLoad of uniqueX) {
    const stepLoad = { type: 'point', x: xLoad, value: -1.0 };
    simConfig.globalLoads.push(stepLoad);

    try {
        const res = analyzeBeam(simConfig);
        let val = 0;
        
        if (target.type === 'reaction') {
            const rKey = Object.keys(res.reactions).find(k => Math.abs(parseFloat(k) - target.location) < TOLERANCE);
            if (rKey) {
                const comp = target.component || 'Fy';
                val = res.reactions[parseFloat(rKey)][comp] || 0;
            }
        } else {
            const arr = getResultArray(res, target.type);
            val = getExactValueAt(res.diagrams.x, arr, target.location);
        }
        
        xData.push(xLoad);
        yData.push(val);

    } catch (err) {
        console.warn(`ILD Analysis failed at x=${xLoad}`);
        xData.push(xLoad);
        yData.push(0);
    }

    simConfig.globalLoads.pop();
  }

  return { x: xData, value: yData };
}

function getResultArray(results: any, type: string) 
{
    if (type === 'shear') return results.diagrams.shear;
    if (type === 'moment') return results.diagrams.moment;
    if (type === 'deflection') return results.diagrams.deflection;
    return [];
}

function getExactValueAt(X: number[], Y: number[], targetX: number) 
{
    if(!X || X.length === 0) return 0;
    const i = X.findIndex(val => val >= targetX - 1e-9);
    
    if (i === -1) return Y[Y.length - 1];
    if (i === 0) return Y[0]; 
    if (i >= X.length) return Y[X.length - 1];

    const x1 = X[i-1], x2 = X[i];
    const y1 = Y[i-1], y2 = Y[i];

    if (Math.abs(x2 - x1) < 1e-9) return y2;

    const ratio = (targetX - x1) / (x2 - x1);
    return y1 + (y2 - y1) * ratio;
}