import { GoogleGenAI, Type, Schema } from "@google/genai";
import { AnalysisResult, BeamConfig } from "../types";

export async function generateEngineeringReport(config: BeamConfig, results: AnalysisResult): Promise<string> {
    const apiKey = process.env.API_KEY;
    if (!apiKey) throw new Error("API Key not found");

    const ai = new GoogleGenAI({ apiKey });
    
    // Simplify data for token efficiency
    const simpleResults = {
        maxShear: results.summary.maxShear,
        maxMoment: results.summary.maxMoment,
        maxDeflection: results.summary.maxDeflection,
        reactions: results.reactions
    };

    const prompt = `
    You are a senior structural engineer. Analyze the following Beam Calculation results.
    
    **Beam Configuration:**
    Length: ${config.length}m
    Supports: ${config.supports.map(s => `${s.type} at ${s.x}m`).join(', ')}
    Loads: ${config.loads.map(l => `${l.type} ${l.value} at ${l.x ?? l.start}`).join(', ')}
    
    **Analysis Results:**
    ${JSON.stringify(simpleResults, null, 2)}
    
    **Task:**
    1. Provide a brief executive summary of the beam's behavior.
    2. Comment on the locations of maximum shear and moment. Are they where you expect?
    3. Suggest any structural concerns (e.g. high deflection, large reactions).
    4. Format the output in clean Markdown. Use bolding for key figures.
    `;

    try {
        const response = await ai.models.generateContent({
            model: 'gemini-3-flash-preview',
            contents: prompt
        });
        return response.text || "No analysis generated.";
    } catch (error) {
        console.error("Gemini Error", error);
        return "Failed to generate AI report. Please check your API configuration.";
    }
}