import React from 'react';
import Chart from 'react-apexcharts';
import { AnalysisResult, BeamConfig } from '../types';
import { ApexOptions } from 'apexcharts';

interface Props {
    results: AnalysisResult;
    config: BeamConfig;
}

const ResultsCharts: React.FC<Props> = ({ results }) => {
    const { x, shear, moment, deflection, bendingStress } = results.diagrams;
    if (!x || !shear || !moment || !deflection) return null;

    // Helper to generate chart options
    const getChartOptions = (color: string, title: string, unit: string, reverseY: boolean): ApexOptions => ({
        chart: {
            type: 'area',
            height: '100%',
            fontFamily: 'inherit',
            toolbar: {
                show: true,
                tools: {
                    download: false, // We use HTML2Canvas for the whole page
                    selection: true,
                    zoom: true,
                    zoomin: true,
                    zoomout: true,
                    pan: true,
                    reset: true
                }
            },
            animations: {
                enabled: false // Disable to improve perf on dense data
            }
        },
        colors: [color],
        dataLabels: {
            enabled: false
        },
        stroke: {
            curve: 'straight',
            width: 2
        },
        fill: {
            type: 'solid',
            opacity: 0.1
        },
        xaxis: {
            type: 'numeric',
            tickAmount: 10,
            decimalsInFloat: 2,
            title: {
                text: 'Distance (m)',
                style: { fontSize: '11px', fontWeight: 600, color: '#64748b' }
            },
            labels: {
                formatter: (value) => parseFloat(String(value)).toFixed(1)
            },
            tooltip: {
                enabled: false // Hide default x-axis tooltip since we have crosshairs
            },
            crosshairs: {
                show: true,
                stroke: { color: '#94a3b8', dashArray: 4 }
            }
        },
        yaxis: {
            reversed: reverseY,
            decimalsInFloat: 2,
            labels: {
                style: { colors: '#64748b' }
            }
        },
        grid: {
            borderColor: '#f1f5f9',
            strokeDashArray: 3,
            xaxis: { lines: { show: true } },
            yaxis: { lines: { show: true } }
        },
        tooltip: {
            custom: ({ series, seriesIndex, dataPointIndex, w }: any) => {
                const xVal = w.globals.seriesX[seriesIndex][dataPointIndex];
                const yVal = series[seriesIndex][dataPointIndex];
                return `<div style="padding:6px 10px;font-size:12px;line-height:1.4">
                    <div style="color:#64748b;font-weight:600">x = ${xVal.toFixed(2)}m</div>
                    <div style="color:#0f172a;font-weight:700">${yVal.toFixed(3)} ${unit}</div>
                </div>`;
            }
        }
    });

    // Data format required for ApexCharts xy numeric timeline: [[x1, y1], [x2, y2], ...]
    const generateSeriesData = (yData: number[], scale = 1) => {
        return x.map((xVal, i) => [xVal, yData[i] * scale]);
    };

    const renderChart = (title: string, dataKey: string, dataArray: number[], color: string, unit: string, scale: number = 1, reverseAxis: boolean = false) => {
        const series = [{
            name: title.split(' (')[0], // Extract just the name without the unit
            data: generateSeriesData(dataArray, scale)
        }];

        return (
            <div className="bg-white p-6 rounded-2xl shadow-sm border border-slate-200">
                <h3 className="text-sm font-bold text-slate-800 mb-2 uppercase tracking-wider">{title}</h3>
                <div className="h-[300px] w-full">
                    <Chart
                        options={getChartOptions(color, title, unit, reverseAxis)}
                        series={series}
                        type="area"
                        height="100%"
                        width="100%"
                    />
                </div>
            </div>
        );
    };

    return (
        <div className="flex flex-col gap-8">
            {renderChart("Shear Force (kN)", "shear", shear, "#059669", "kN")}
            {/* Reverse Axis for Moment: Plots Sagging (+) Downwards (Tension Side) */}
            {renderChart("Bending Moment (kNm)", "moment", moment, "#2563eb", "kNm", 1, true)}
            {renderChart("Deflection (mm)", "deflection", deflection, "#d97706", "mm", 1000)}
            {renderChart("Bending Stress (MPa)", "stress", bendingStress || [], "#7c3aed", "MPa", 1 / 1000)}
        </div>
    );
};

export default ResultsCharts;