import React, { useState, useEffect } from 'react';
import {
    Activity, Zap, Settings, Plus, Trash2,
    Bot, ChevronRight, Anchor, Scaling, ArrowDownToLine, Move, Minus,
    TrendingUp, Spline, Ruler, CheckCircle2, X
} from 'lucide-react';
import Chart from 'react-apexcharts';
import { ApexOptions } from 'apexcharts';
import { BeamConfig, AnalysisResult, ILDResult, ReactionComponent } from './types';
import { analyzeBeam } from './services/fem-engine-pro';
import { calculateInfluenceLine, ILDTarget } from './services/ild-module';
import { generateEngineeringReport } from './services/geminiService';
import { generatePDFReport } from './services/pdf-generator';
import BeamVisualizer from './components/BeamVisualizer';
import ResultsCharts from './components/ResultsCharts';

// --- ILD CHART COMPONENT (ApexCharts) ---
const ILDChart: React.FC<{ ildResult: ILDResult }> = ({ ildResult }) => {
    const series = [{ name: 'ILD Value', data: ildResult.x.map((x, i) => [x, ildResult.value[i]]) }];
    const options: ApexOptions = {
        chart: { type: 'area', toolbar: { show: false }, animations: { enabled: false }, fontFamily: 'inherit' },
        colors: ['#9333ea'],
        dataLabels: { enabled: false },
        stroke: { curve: 'straight', width: 2 },
        fill: { type: 'solid', opacity: 0.1 },
        xaxis: {
            type: 'numeric', decimalsInFloat: 2,
            labels: { formatter: (v) => parseFloat(String(v)).toFixed(1) },
            crosshairs: { show: true, stroke: { color: '#94a3b8', dashArray: 4 } },
            tooltip: { enabled: false }
        },
        yaxis: { decimalsInFloat: 3, labels: { style: { colors: '#64748b' } } },
        grid: { borderColor: '#f1f5f9', strokeDashArray: 3 },
        tooltip: {
            theme: 'light',
            x: { formatter: (val: number) => `x = ${val.toFixed(2)}m` },
            y: { formatter: (val: number) => `${val.toFixed(3)}` }
        }
    };
    return <Chart options={options} series={series} type="area" height="100%" width="100%" />;
};

// --- INITIAL CONFIGURATIONS ---

const getInitialBeamConfig = (): BeamConfig => ({
    length: 10,
    E: 200000000,
    I: 0.0004,
    depth: 0.5,
    supports: [
        { id: 's1', type: 'pin', x: 0 },
        { id: 's2', type: 'roller', x: 10 }
    ],
    loads: [],
    hinges: [],
    sections: []
});

const Label = ({ children, info }: { children?: React.ReactNode, info?: string }) => (
    <div className="flex justify-between items-center mb-1.5 md:mb-1.5">
        <label className="text-[12px] md:text-[11px] font-semibold text-slate-300 uppercase tracking-wide">{children}</label>
        {info && <span className="text-[11px] md:text-[10px] text-slate-500">{info}</span>}
    </div>
);

// Safe math evaluator using Function constructor with Math object scope
const safeEvaluateMath = (expr: string): number | null => {
    try {
        // Remove spaces and everything except numbers, operators, and common Math functions/constants
        const sanitized = expr.toLowerCase().replace(/[^0-9+\-*/().e pi sin cos tan sqrt abs \^]/g, '');
        if (!sanitized) return null;

        // Convert 'pi' to 'Math.PI', 'sin' to 'Math.sin' etc.
        let finalExpr = sanitized
            .replace(/pi/g, 'Math.PI')
            .replace(/sin/g, 'Math.sin')
            .replace(/cos/g, 'Math.cos')
            .replace(/tan/g, 'Math.tan')
            .replace(/sqrt/g, 'Math.sqrt')
            .replace(/abs/g, 'Math.abs')
            .replace(/\^/g, '**'); // Convert ^ to exponent operator

        // eslint-disable-next-line no-new-func
        const result = new Function(`return ${finalExpr}`)();

        if (typeof result === 'number' && !isNaN(result) && isFinite(result)) {
            // Fix floating point precision artifacts (e.g. 5 * 10^-6 = 0.0000049999...)
            return parseFloat(result.toPrecision(12));
        }
        return null;
    } catch (e) {
        return null;
    }
};

const DarkInput = ({ value, onChange, className, suffix, ...props }: any) => {
    const [localVal, setLocalVal] = useState(value?.toString() ?? "");
    const [isFocused, setIsFocused] = useState(false);

    useEffect(() => {
        if (!isFocused) setLocalVal(value?.toString() ?? "");
    }, [value, isFocused]);

    const handleEval = () => {
        const result = safeEvaluateMath(localVal);
        if (result !== null) {
            setLocalVal(result.toString());
            onChange(result);
        } else if (localVal === '' || localVal === '-') {
            onChange(0);
        } else {
            // Revert on invalid math
            setLocalVal(value?.toString() ?? "");
        }
    };

    return (
        <div className="relative flex items-center group">
            <input
                type="text"
                value={localVal}
                onFocus={() => setIsFocused(true)}
                onBlur={() => {
                    setIsFocused(false);
                    handleEval();
                }}
                onKeyDown={(e) => {
                    if (e.key === 'Enter') {
                        e.currentTarget.blur();
                    }
                }}
                onChange={(e) => setLocalVal(e.target.value)}
                className={`bg-slate-50 border border-slate-700 text-slate-900 text-base md:text-sm rounded pl-3 py-2.5 md:py-2 outline-none w-full font-mono transition-all focus:ring-2 focus:ring-blue-500 focus:border-blue-500 ${suffix ? 'pr-20' : 'pr-6'} ${className}`}
                autoComplete="off"
                title="Enter numbers or math forms (e.g. '10/2', 'sin(45)')"
                {...props}
            />
            {suffix && (
                <div className="absolute right-1 top-1 bottom-1 px-2 flex items-center bg-slate-100 rounded text-[11px] md:text-xs text-slate-500 font-medium pointer-events-none border-l border-slate-200">
                    {suffix}
                </div>
            )}
            <div className={`absolute pointer-events-none transition-opacity duration-200 ${isFocused ? 'opacity-100' : 'opacity-0'} ${suffix ? 'right-12 md:right-14' : 'right-2'} text-[10px] text-blue-400 font-bold bg-white px-1`}>
                =
            </div>
        </div>
    );
};

const DarkSelect = ({ className, children, ...props }: any) => (
    <div className="relative">
        <select
            className={`appearance-none bg-white border border-slate-300 text-slate-900 text-base md:text-sm rounded px-3 py-2.5 md:py-2 pr-8 focus:ring-2 focus:ring-blue-500 outline-none w-full transition-all font-medium ${className}`}
            {...props}
        >
            {children}
        </select>
        <div className="pointer-events-none absolute inset-y-0 right-0 flex items-center px-3 text-slate-400">
            <ChevronRight size={14} className="rotate-90" />
        </div>
    </div>
);

const PrimaryButton = ({ onClick, children, icon: Icon, className }: any) => (
    <button
        onClick={onClick}
        className={`w-full py-3 md:py-2.5 bg-blue-600 hover:bg-blue-500 text-white text-sm md:text-xs font-bold rounded flex items-center justify-center gap-2 transition-all shadow-lg shadow-blue-900/20 active:translate-y-0.5 ${className}`}
    >
        {Icon && <Icon size={16} className="md:w-[14px] md:h-[14px]" />}
        {children}
    </button>
);

const InspectorHeader = ({ title, icon: Icon }: { title: string, icon?: any }) => (
    <div className="flex items-center gap-2 mb-6 pb-4 border-b border-slate-800">
        {Icon && <div className="p-1.5 bg-slate-800 rounded text-blue-400"><Icon size={16} /></div>}
        <h2 className="text-slate-100 font-bold text-sm tracking-wide">{title}</h2>
    </div>
);

const getNextId = (list: any[]) => {
    const numericIds = list.map(item => {
        const match = item.id.match(/(\d+)$/);
        return match ? parseInt(match[0], 10) : NaN;
    }).filter(n => !isNaN(n));
    return numericIds.length > 0 ? (Math.max(...numericIds) + 1).toString() : "1";
};

export default function App() {
    const [activeTool, setActiveTool] = useState<'hinges' | 'properties' | 'supports' | 'loads' | 'results' | 'ild'>('properties');
    const [selectedId, setSelectedId] = useState<string | null>(null);
    const [isInspectorOpen, setIsInspectorOpen] = useState(false);

    const handleToolSelect = (tool: typeof activeTool) => {
        setActiveTool(tool);
        setIsInspectorOpen(true);
    };

    const [beamConfig, setBeamConfig] = useState<BeamConfig>(getInitialBeamConfig());
    const [beamResults, setBeamResults] = useState<AnalysisResult | null>(null);
    const [ildTarget, setIldTarget] = useState<ILDTarget>({ type: 'shear', location: 2, component: 'Fy' });
    const [ildResult, setIldResult] = useState<ILDResult | null>(null);

    // Viewport State
    const [viewConfig, setViewConfig] = useState({ zoom: 1, pan: { x: 0, y: 0 } });

    const [isPdfLoading, setIsPdfLoading] = useState(false);

    useEffect(() => { setSelectedId(null); }, [activeTool]);

    useEffect(() => {
        try {
            setBeamResults(analyzeBeam(beamConfig));
            if (activeTool === 'ild') setIldResult(calculateInfluenceLine(beamConfig, ildTarget));
        } catch (e) {
            setBeamResults(null);
        }
    }, [beamConfig, ildTarget, activeTool]);

    const handleDelete = (id: string, listName: 'loads' | 'supports' | 'hinges' | 'sections') => {
        setBeamConfig(prev => ({ ...prev, [listName]: (prev as any)[listName].filter((x: any) => x.id !== id) }));
        setSelectedId(null);
    };

    // --- VIEW CONTROLS ---
    const handleZoomIn = () => setViewConfig(p => ({ ...p, zoom: Math.min(5, p.zoom * 1.2) }));
    const handleZoomOut = () => setViewConfig(p => ({ ...p, zoom: Math.max(0.2, p.zoom / 1.2) }));
    const handleFit = () => setViewConfig({ zoom: 1, pan: { x: 0, y: 0 } });

    // --- RENDERERS ---
    const renderHingeInspector = () => {
        const hinges = beamConfig.hinges || [];
        const selectedHinge = hinges.find(x => x.id === selectedId);
        return (
            <div className="space-y-6">
                <InspectorHeader title="Internal Hinges" icon={Spline} />
                <div className="space-y-4">
                    <Label>Hinge Management</Label>
                    {selectedHinge ? (
                        <div className="bg-slate-800 p-4 rounded-lg border border-slate-700 space-y-4">
                            <div className="flex justify-between items-center border-b border-slate-700 pb-2">
                                <span className="text-white font-mono font-bold">{selectedHinge.id}</span>
                                <button onClick={() => handleDelete(selectedHinge.id, 'hinges')} className="text-red-400 hover:text-red-300"><Trash2 size={16} /></button>
                            </div>
                            <div className="space-y-1">
                                <div className="text-[10px] text-slate-400 uppercase font-bold">Location (x)</div>
                                <DarkInput value={selectedHinge.x} onChange={(v: number) => setBeamConfig(p => ({ ...p, hinges: p.hinges.map(x => x.id === selectedHinge.id ? { ...x, x: v } : x) }))} suffix="m" />
                            </div>
                            <button onClick={() => setSelectedId(null)} className="w-full py-2 mt-2 text-xs font-bold text-slate-400 hover:text-white border border-slate-600 rounded hover:bg-slate-700 transition-colors">Done Editing</button>
                        </div>
                    ) : (
                        <PrimaryButton icon={Plus} onClick={() => { const newId = `h${getNextId(hinges as any[])}`; setBeamConfig(p => ({ ...p, hinges: [...(p.hinges || []), { id: newId, x: p.length / 2 }] })); setSelectedId(newId); }}>Add Internal Hinge</PrimaryButton>
                    )}
                </div>
                {!selectedHinge && hinges.length > 0 && (
                    <div className="pt-4 border-t border-slate-800"><Label>Existing Hinges</Label><div className="flex flex-col gap-2 mt-2">{hinges.map(h => (<button key={h.id} onClick={() => setSelectedId(h.id)} className="flex items-center justify-between px-3 py-2 bg-slate-800 hover:bg-slate-700 rounded border border-slate-700 transition-colors text-left"><span className="text-xs text-slate-300 font-mono">{h.id}</span><span className="text-xs text-blue-400 font-bold">x = {h.x}m</span></button>))}</div></div>
                )}
            </div>
        );
    };

    const renderBeamInspector = () => {
        const sections = beamConfig.sections || [];
        return (
            <div className="space-y-6">
                <InspectorHeader title="Beam Properties" icon={Ruler} />
                <div className="space-y-4">
                    <Label>Global Geometry</Label>
                    <DarkInput value={beamConfig.length} onChange={(v: number) => setBeamConfig(p => ({ ...p, length: Math.max(0.1, v) }))} suffix="m" />
                    <div className="text-[10px] text-slate-500">Total length of the continuous beam.</div>
                </div>
                <div className="pt-4 border-t border-slate-800 space-y-4">
                    <Label>Default Section Properties</Label>
                    <div className="grid grid-cols-2 gap-3">
                        <div><div className="text-[10px] text-slate-400 mb-1">Modulus (E)</div><DarkInput value={beamConfig.E} onChange={(v: number) => setBeamConfig(p => ({ ...p, E: v }))} suffix="kPa" /></div>
                        <div><div className="text-[10px] text-slate-400 mb-1">Inertia (I)</div><DarkInput value={beamConfig.I} onChange={(v: number) => setBeamConfig(p => ({ ...p, I: v }))} suffix="m⁴" /></div>
                        <div className="col-span-2">
                            <div className="text-[10px] text-slate-400 mb-1">Depth (d)</div>
                            <DarkInput value={beamConfig.depth} onChange={(v: number) => setBeamConfig(p => ({ ...p, depth: v }))} suffix="m" />
                            <div className="mt-2 p-2.5 bg-slate-800/50 border border-slate-700/50 rounded flex gap-2 items-start text-[10px] leading-relaxed text-slate-400">
                                <Activity size={12} className="text-blue-400 shrink-0 mt-0.5" />
                                <div>
                                    <span className="text-slate-300 font-bold">Why input I and d separately?</span><br />
                                    For complex shapes (like steel I-beams), Inertia (I) isn't just <span className="font-mono text-slate-500">bd³/12</span>. Depth (d) is used exclusively to find the extreme fiber bending stress <span className="font-mono text-slate-500">(σ = M·c/I)</span>.
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div className="pt-4 border-t border-slate-800">
                    <div className="flex justify-between items-center mb-3"><Label>Variable Sections</Label><button onClick={() => setBeamConfig(p => ({ ...p, sections: [...(p.sections || []), { x: 0, end: p.length / 2, E: p.E, I: p.I * 2, depth: (p.depth || 0.5) * 1.2 }] }))} className="p-1 bg-slate-800 rounded text-blue-400 hover:bg-slate-700 transition-colors" title="Add Section Segment"><Plus size={14} /></button></div>
                    <div className="space-y-3">
                        {sections.map((sec, i) => (
                            <div key={i} className="bg-slate-800 p-3 rounded border border-slate-700 text-xs space-y-2 relative group">
                                <button onClick={() => setBeamConfig(p => ({ ...p, sections: p.sections?.filter((_, idx) => idx !== i) }))} className="absolute top-2 right-2 text-slate-600 hover:text-red-400"><Trash2 size={14} /></button>
                                <div className="font-bold text-slate-300">Segment {i + 1}</div>
                                <div className="grid grid-cols-2 gap-2">
                                    <DarkInput value={sec.x} onChange={(v: number) => setBeamConfig(p => ({ ...p, sections: p.sections?.map((s, idx) => idx === i ? { ...s, x: v } : s) }))} suffix="start m" />
                                    <DarkInput value={sec.end} onChange={(v: number) => setBeamConfig(p => ({ ...p, sections: p.sections?.map((s, idx) => idx === i ? { ...s, end: v } : s) }))} suffix="end m" />
                                    <DarkInput value={sec.E} onChange={(v: number) => setBeamConfig(p => ({ ...p, sections: p.sections?.map((s, idx) => idx === i ? { ...s, E: v } : s) }))} suffix="E (kPa)" />
                                    <DarkInput value={sec.I} onChange={(v: number) => setBeamConfig(p => ({ ...p, sections: p.sections?.map((s, idx) => idx === i ? { ...s, I: v } : s) }))} suffix="I (m⁴)" />
                                    <div className="col-span-2">
                                        <DarkInput value={sec.depth} onChange={(v: number) => setBeamConfig(p => ({ ...p, sections: p.sections?.map((s, idx) => idx === i ? { ...s, depth: v } : s) }))} suffix="d (m)" />
                                    </div>
                                </div>
                            </div>
                        ))}
                        {sections.length === 0 && <div className="text-xs text-slate-500 italic text-center py-2">Beam is prismatic (uniform section).</div>}
                    </div>
                </div>
            </div>
        );
    };

    const renderSupportInspector = () => {
        const supports = beamConfig.supports;
        const selectedSupport = supports.find(s => s.id === selectedId);

        if (selectedSupport) {
            return (
                <div className="space-y-6">
                    <div className="flex items-center justify-between pb-4 border-b border-slate-800">
                        <div><Label>Editing Support</Label><div className="text-xl font-bold text-white font-mono">{selectedSupport.id}</div></div>
                        <button onClick={() => handleDelete(selectedSupport.id, 'supports')} className="p-2 bg-red-900/20 text-red-400 rounded hover:bg-red-900/40"><Trash2 size={18} /></button>
                    </div>
                    <div className="space-y-4">
                        <div><Label>Type</Label>
                            <div className="grid grid-cols-2 gap-2">
                                {['pin', 'roller', 'fixed', 'spring_y'].map(t => (
                                    <button key={t} onClick={() => setBeamConfig(p => ({ ...p, supports: p.supports.map(s => s.id === selectedSupport.id ? { ...s, type: t as any } : s) }))} className={`px-2 py-2 text-xs font-bold rounded border ${selectedSupport.type === t ? 'bg-blue-600 border-blue-500 text-white' : 'bg-slate-800 border-slate-700 text-slate-400 hover:bg-slate-700'}`}>{t}</button>
                                ))}
                            </div>
                        </div>
                        <div><Label>Position (x)</Label><DarkInput value={selectedSupport.x} onChange={(v: number) => setBeamConfig(p => ({ ...p, supports: p.supports.map(s => s.id === selectedSupport.id ? { ...s, x: v } : s) }))} suffix="m" /></div>
                        {selectedSupport.type === 'spring_y' && (
                            <div><Label>Stiffness (k)</Label><DarkInput value={selectedSupport.stiffness} onChange={(v: number) => setBeamConfig(p => ({ ...p, supports: p.supports.map(s => s.id === selectedSupport.id ? { ...s, stiffness: v } : s) }))} suffix="kN/m" /></div>
                        )}
                        <div><Label>Settlement (dy)</Label><DarkInput value={selectedSupport.settlement} onChange={(v: number) => setBeamConfig(p => ({ ...p, supports: p.supports.map(s => s.id === selectedSupport.id ? { ...s, settlement: v } : s) }))} suffix="m" /></div>
                    </div>
                    <button onClick={() => setSelectedId(null)} className="w-full py-2 mt-4 text-xs font-bold text-slate-400 hover:text-white border border-slate-700 rounded hover:bg-slate-800 transition-colors">Done</button>
                </div>
            );
        }

        return (
            <div className="space-y-6">
                <InspectorHeader title="Supports" icon={Anchor} />
                <div className="grid grid-cols-2 gap-2">
                    {['pin', 'roller', 'fixed', 'spring_y'].map(type => (
                        <button key={type} onClick={() => {
                            const newId = `s${getNextId(supports)}`;
                            setBeamConfig(p => ({ ...p, supports: [...p.supports, { id: newId, type: type as any, x: p.length / 2 }] }));
                            setSelectedId(newId);
                        }} className="flex flex-col items-center justify-center p-3 bg-slate-800 border border-slate-700 rounded hover:bg-slate-700 hover:border-slate-500 transition-all group">
                            <span className="text-xs font-bold text-slate-300 group-hover:text-white uppercase">{type}</span>
                        </button>
                    ))}
                </div>
                <div className="pt-4 border-t border-slate-800">
                    <Label>Existing Supports</Label>
                    <div className="flex flex-col gap-2 mt-2">
                        {supports.map(s => (
                            <button key={s.id} onClick={() => setSelectedId(s.id)} className="flex items-center justify-between px-3 py-2 bg-slate-800 hover:bg-slate-700 rounded border border-slate-700 transition-colors text-left">
                                <span className="text-xs text-slate-300 font-mono">{s.id} <span className="text-slate-500">({s.type})</span></span>
                                <span className="text-xs text-blue-400 font-bold">x = {s.x}m</span>
                            </button>
                        ))}
                    </div>
                </div>
            </div>
        )
    };

    const renderLoadInspector = () => {
        const loads = beamConfig.loads;
        const selectedLoad = loads.find(l => l.id === selectedId);

        if (selectedLoad) {
            return (
                <div className="space-y-6">
                    <div className="flex items-center justify-between pb-4 border-b border-slate-800">
                        <div><Label>Editing Load</Label><div className="text-xl font-bold text-white font-mono">{selectedLoad.id}</div></div>
                        <button onClick={() => handleDelete(selectedLoad.id, 'loads')} className="p-2 bg-red-900/20 text-red-400 rounded hover:bg-red-900/40"><Trash2 size={18} /></button>
                    </div>
                    <div className="space-y-4">
                        <div><Label>Load Type</Label><div className="text-xs font-bold text-blue-400 uppercase">{selectedLoad.type}</div></div>
                        {(selectedLoad.type === 'point' || selectedLoad.type === 'moment') && (
                            <div><Label>Position (x)</Label><DarkInput value={selectedLoad.x} onChange={(v: number) => setBeamConfig(p => ({ ...p, loads: p.loads.map(l => l.id === selectedLoad.id ? { ...l, x: v } : l) }))} suffix="m" /></div>
                        )}
                        {(selectedLoad.type === 'udl' || selectedLoad.type === 'uvl') && (
                            <div className="grid grid-cols-2 gap-2">
                                <div><Label>Start</Label><DarkInput value={selectedLoad.start} onChange={(v: number) => setBeamConfig(p => ({ ...p, loads: p.loads.map(l => l.id === selectedLoad.id ? { ...l, start: v } : l) }))} suffix="m" /></div>
                                <div><Label>End</Label><DarkInput value={selectedLoad.end} onChange={(v: number) => setBeamConfig(p => ({ ...p, loads: p.loads.map(l => l.id === selectedLoad.id ? { ...l, end: v } : l) }))} suffix="m" /></div>
                            </div>
                        )}
                        <div><Label>Magnitude</Label>
                            {selectedLoad.type === 'uvl' ? (
                                <div className="grid grid-cols-2 gap-2">
                                    <DarkInput value={selectedLoad.valueStart} onChange={(v: number) => setBeamConfig(p => ({ ...p, loads: p.loads.map(l => l.id === selectedLoad.id ? { ...l, valueStart: v } : l) }))} suffix="Start" />
                                    <DarkInput value={selectedLoad.valueEnd} onChange={(v: number) => setBeamConfig(p => ({ ...p, loads: p.loads.map(l => l.id === selectedLoad.id ? { ...l, valueEnd: v } : l) }))} suffix="End" />
                                </div>
                            ) : (
                                <DarkInput value={selectedLoad.value} onChange={(v: number) => setBeamConfig(p => ({ ...p, loads: p.loads.map(l => l.id === selectedLoad.id ? { ...l, value: v } : l) }))} suffix={selectedLoad.type === 'moment' ? 'kNm' : 'kN'} />
                            )}
                        </div>
                    </div>
                    <button onClick={() => setSelectedId(null)} className="w-full py-2 mt-4 text-xs font-bold text-slate-400 hover:text-white border border-slate-700 rounded hover:bg-slate-800 transition-colors">Done</button>
                </div>
            );
        }

        return (
            <div className="space-y-6">
                <InspectorHeader title="Loads" icon={ArrowDownToLine} />
                <div className="grid grid-cols-2 gap-2">
                    {['point', 'moment', 'udl', 'uvl'].map(type => (
                        <button key={type} onClick={() => {
                            const newId = `l${getNextId(loads)}`;
                            let newLoad: any = { id: newId, type, value: -10 };
                            if (type === 'point' || type === 'moment') newLoad.x = beamConfig.length / 2;
                            if (type === 'udl') { newLoad.start = 0; newLoad.end = beamConfig.length; newLoad.value = -5; }
                            if (type === 'uvl') { newLoad.start = 0; newLoad.end = beamConfig.length; newLoad.valueStart = -5; newLoad.valueEnd = -10; }
                            setBeamConfig(p => ({ ...p, loads: [...p.loads, newLoad] }));
                            setSelectedId(newId);
                        }} className="flex flex-col items-center justify-center p-3 bg-slate-800 border border-slate-700 rounded hover:bg-slate-700 hover:border-slate-500 transition-all group">
                            <span className="text-xs font-bold text-slate-300 group-hover:text-white uppercase">{type}</span>
                        </button>
                    ))}
                </div>
                <div className="pt-4 border-t border-slate-800">
                    <Label>Existing Loads</Label>
                    <div className="flex flex-col gap-2 mt-2">
                        {loads.map(l => (
                            <button key={l.id} onClick={() => setSelectedId(l.id)} className="flex items-center justify-between px-3 py-2 bg-slate-800 hover:bg-slate-700 rounded border border-slate-700 transition-colors text-left">
                                <span className="text-xs text-slate-300 font-mono">{l.id} <span className="text-slate-500">({l.type})</span></span>
                                <span className="text-xs text-blue-400 font-bold">{l.type.includes('u') ? `${l.start}-${l.end}m` : `@${l.x}m`}</span>
                            </button>
                        ))}
                    </div>
                </div>
            </div>
        )
    };

    const renderILDPanel = () => {
        return (
            <div className="space-y-6">
                <InspectorHeader title="Influence Lines" icon={TrendingUp} />
                <div className="space-y-4">
                    <div><Label>Target Response</Label>
                        <DarkSelect value={ildTarget.type} onChange={(e: any) => setIldTarget(p => ({ ...p, type: e.target.value }))}>
                            <option value="reaction">Reaction</option>
                            <option value="shear">Shear Force</option>
                            <option value="moment">Bending Moment</option>
                            <option value="deflection">Deflection</option>
                        </DarkSelect>
                    </div>
                    <div><Label>Location (x)</Label>
                        <DarkInput value={ildTarget.location} onChange={(v: number) => setIldTarget(p => ({ ...p, location: v }))} suffix="m" />
                    </div>
                    {ildTarget.type === 'reaction' && (
                        <div><Label>Component</Label>
                            <DarkSelect value={ildTarget.component} onChange={(e: any) => setIldTarget(p => ({ ...p, component: e.target.value }))}>
                                <option value="Fy">Vertical (Fy)</option>
                                <option value="Mz">Moment (Mz)</option>
                            </DarkSelect>
                        </div>
                    )}
                    <div className="text-[10px] text-slate-400 p-2 bg-slate-800 rounded border border-slate-700">
                        Click on the beam diagram to pick location.
                    </div>
                </div>
            </div>
        );
    };

    const renderResultsPanel = () => {
        return (
            <div className="space-y-6">
                <InspectorHeader title="Analysis Results" icon={Zap} />
                {beamResults && (
                    <div className="space-y-4 pt-4 border-t border-slate-800">
                        <Label>Detailed Report</Label>
                        <PrimaryButton
                            icon={Bot}
                            onClick={async () => {
                                setIsPdfLoading(true);
                                await generatePDFReport(beamConfig, beamResults, 'printable-visualizer', 'printable-charts');
                                setIsPdfLoading(false);
                            }}
                        >
                            {isPdfLoading ? 'Generating PDF...' : 'Download PDF Report'}
                        </PrimaryButton>
                    </div>
                )}
            </div>
        );
    };

    return (
        <div className="flex flex-col md:flex-row h-screen bg-slate-900 font-sans text-slate-100 overflow-hidden">
            {/* Sidebar / Bottom Nav */}
            <div className="w-full md:w-16 h-16 md:h-screen bg-slate-950 border-t md:border-t-0 md:border-r border-slate-800 flex flex-row md:flex-col items-center justify-around md:justify-start px-2 md:py-4 z-50 order-last md:order-first shrink-0 shadow-[0_-4px_10px_rgba(0,0,0,0.3)] md:shadow-none">
                <div className="hidden md:flex mb-6"><div className="w-10 h-10 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-lg flex items-center justify-center shadow-lg shadow-blue-900/40"><Activity size={20} className="text-white" /></div></div>
                <div className="flex flex-row md:flex-col items-center w-full md:w-auto justify-around md:justify-start flex-1 md:flex-none md:space-y-4">
                    <button onClick={() => handleToolSelect('properties')} className={`p-2.5 md:p-3 rounded-xl transition-all ${activeTool === 'properties' ? 'bg-blue-600 text-white shadow-lg' : 'text-slate-500 hover:text-slate-300 hover:bg-slate-800'}`} title="Beam Properties"><Ruler size={20} /></button>
                    <button onClick={() => handleToolSelect('hinges')} className={`p-2.5 md:p-3 rounded-xl transition-all ${activeTool === 'hinges' ? 'bg-blue-600 text-white shadow-lg' : 'text-slate-500 hover:text-slate-300 hover:bg-slate-800'}`} title="Internal Hinges"><Spline size={20} /></button>
                    <button onClick={() => handleToolSelect('supports')} className={`p-2.5 md:p-3 rounded-xl transition-all ${activeTool === 'supports' ? 'bg-blue-600 text-white shadow-lg' : 'text-slate-500 hover:text-slate-300 hover:bg-slate-800'}`} title="Supports"><Anchor size={20} /></button>
                    <button onClick={() => handleToolSelect('loads')} className={`p-2.5 md:p-3 rounded-xl transition-all ${activeTool === 'loads' ? 'bg-blue-600 text-white shadow-lg' : 'text-slate-500 hover:text-slate-300 hover:bg-slate-800'}`} title="Loads"><ArrowDownToLine size={20} /></button>
                    <button onClick={() => handleToolSelect('ild')} className={`p-2.5 md:p-3 rounded-xl transition-all ${activeTool === 'ild' ? 'bg-purple-600 text-white shadow-lg' : 'text-slate-500 hover:text-slate-300 hover:bg-slate-800'}`} title="Influence Lines"><TrendingUp size={20} /></button>
                    <div className="hidden md:block w-8 h-px bg-slate-800 my-2"></div>
                    <button onClick={() => handleToolSelect('results')} className={`p-2.5 md:p-3 rounded-xl transition-all ${activeTool === 'results' ? 'bg-emerald-600 text-white shadow-lg' : 'text-slate-500 hover:text-slate-300 hover:bg-slate-800'}`} title="Analysis Results"><Zap size={20} /></button>
                </div>
                <div className="hidden md:flex mt-auto flex-col items-center gap-4"><button className="text-slate-500 hover:text-white p-2"><Settings size={20} /></button></div>
            </div>

            {/* Backdrop overlay for mobile */}
            {isInspectorOpen && (
                <div
                    className="fixed inset-0 bg-black/60 z-30 md:hidden backdrop-blur-sm transition-opacity"
                    onClick={() => setIsInspectorOpen(false)}
                />
            )}

            {/* Inspector Drawer */}
            <div className={`fixed md:relative inset-y-0 left-0 w-[85vw] sm:w-80 md:w-80 bg-slate-900 border-r border-slate-800 flex flex-col z-40 shadow-2xl transition-transform duration-300 ease-in-out pb-16 md:pb-0 ${isInspectorOpen ? 'translate-x-0' : '-translate-x-full'} md:translate-x-0`}>
                <div className="h-14 border-b border-slate-800 flex items-center justify-between px-6 shrink-0">
                    <span className="font-bold text-slate-100 uppercase tracking-wide text-xs flex items-center gap-2 flex-wrap">Beam Mode<ChevronRight size={12} className="shrink-0" /><span className="text-blue-400">{activeTool}</span></span>
                    <button onClick={() => setIsInspectorOpen(false)} className="md:hidden p-2 text-slate-400 hover:text-white hover:bg-slate-800 rounded-lg transition-colors"><X size={18} /></button>
                </div>
                <div className="flex-1 p-4 md:p-6 overflow-y-auto">
                    {activeTool === 'properties' && renderBeamInspector()}
                    {activeTool === 'hinges' && renderHingeInspector()}
                    {activeTool === 'supports' && renderSupportInspector()}
                    {activeTool === 'loads' && renderLoadInspector()}
                    {activeTool === 'ild' && renderILDPanel()}
                    {activeTool === 'results' && renderResultsPanel()}
                </div>
            </div>

            <div className="flex-1 relative bg-slate-50 flex flex-col overflow-hidden h-[calc(100vh-4rem)] md:h-screen w-full">
                {/* Mobile header (when drawer is closed) */}
                <div className="md:hidden h-16 bg-white border-b border-slate-200 flex items-center justify-between px-4 sm:px-6 shrink-0 shadow-sm z-10 w-full">
                    <div className="flex items-center gap-2 text-slate-800 font-bold tracking-wide text-lg">
                        <Activity size={24} className="text-blue-600" />
                        <span>CivilBeam</span>
                    </div>
                    <button onClick={() => setIsInspectorOpen(true)} className="px-4 py-2 bg-slate-100 text-slate-600 rounded-lg hover:bg-slate-200 text-sm font-bold uppercase flex items-center gap-1 active:bg-slate-300 transition-colors">
                        Edit <ChevronRight size={16} />
                    </button>
                </div>

                {activeTool === 'results' ? (
                    <div className="w-full h-full overflow-y-auto flex flex-col pb-20 md:pb-0">
                        <div className="w-full h-[40vh] md:h-[60vh] min-h-[300px] md:min-h-[500px] bg-slate-100 relative border-b border-slate-200 shrink-0 shadow-md z-0">
                            <div className="absolute inset-0 p-3 md:p-6 flex items-center justify-center">
                                <div id="printable-visualizer" className="relative w-full h-full bg-white rounded-xl shadow-lg border border-slate-200 overflow-hidden">
                                    <div className="absolute top-2 right-2 md:top-4 md:right-4 z-20 flex flex-col gap-2">
                                        <div className="bg-white p-1 rounded-lg shadow-md border border-slate-200 flex flex-col gap-1">
                                            <button onClick={handleZoomIn} className="p-1.5 md:p-2 text-slate-600 hover:bg-slate-100 rounded" title="Zoom In"><Plus size={16} /></button>
                                            <button onClick={handleZoomOut} className="p-1.5 md:p-2 text-slate-600 hover:bg-slate-100 rounded" title="Zoom Out"><Minus size={16} /></button>
                                            <button onClick={handleFit} className="p-1.5 md:p-2 text-slate-600 hover:bg-slate-100 rounded" title="Fit"><Move size={16} /></button>
                                        </div>
                                    </div>
                                    <div className="w-full h-full p-4 md:p-10 flex items-center justify-center"><div className="w-full aspect-[2/1] min-h-[150px]"><BeamVisualizer config={beamConfig} selectedILDPoint={null} onSelectILDPoint={undefined} zoom={viewConfig.zoom} pan={viewConfig.pan} /></div></div>
                                </div>
                            </div>
                        </div>
                        <div className="flex-1 bg-slate-50 p-4 md:p-8">
                            <div className="max-w-6xl mx-auto">
                                <h2 className="text-lg md:text-xl font-bold text-slate-800 mb-4 md:mb-6 flex items-center gap-2 uppercase tracking-wide">
                                    <Activity size={18} className="text-blue-600" />
                                    Detailed Results
                                </h2>
                                {beamResults ? (
                                    <div id="printable-charts" className="space-y-4 md:space-y-8 bg-slate-50">
                                        <ResultsCharts results={beamResults} config={beamConfig} />
                                    </div>
                                ) : (
                                    <div className="flex flex-col items-center justify-center h-48 md:h-64 text-slate-400 bg-white rounded-xl border border-slate-200 border-dashed m-1">
                                        <div className="p-3 bg-slate-50 rounded-full mb-2"><Activity size={20} className="opacity-50" /></div>
                                        <div className="text-sm font-medium">No results available</div>
                                        <div className="text-[10px] sm:text-xs mt-1">Configure your structure to see analysis.</div>
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                ) : (
                    <div className="w-full h-full flex flex-col relative pb-0 md:pb-0">
                        <div className="absolute top-2 right-2 md:top-4 md:right-4 z-10 flex flex-col gap-2">
                            <div className="bg-white p-1 rounded-lg shadow-md border border-slate-200 flex flex-col gap-1">
                                <button onClick={handleZoomIn} className="p-2.5 md:p-2 text-slate-600 hover:bg-slate-100 rounded active:bg-slate-200" title="Zoom In"><Plus size={20} className="md:w-4 md:h-4" /></button>
                                <button onClick={handleZoomOut} className="p-2.5 md:p-2 text-slate-600 hover:bg-slate-100 rounded active:bg-slate-200" title="Zoom Out"><Minus size={20} className="md:w-4 md:h-4" /></button>
                                <button onClick={handleFit} className="p-2.5 md:p-2 text-slate-600 hover:bg-slate-100 rounded active:bg-slate-200" title="Fit"><Move size={20} className="md:w-4 md:h-4" /></button>
                            </div>
                        </div>
                        <div className="flex-1 relative cursor-crosshair min-h-0 bg-slate-100">
                            <div className="w-full h-full p-4 md:p-10 flex items-center justify-center"><div className="w-full max-w-5xl aspect-[2/1] min-h-[200px]"><BeamVisualizer config={beamConfig} selectedILDPoint={activeTool === 'ild' && ildTarget.type !== 'reaction' ? ildTarget.location : null} onSelectILDPoint={activeTool === 'ild' && ildTarget.type !== 'reaction' ? (x) => setIldTarget(p => ({ ...p, location: x })) : undefined} zoom={viewConfig.zoom} pan={viewConfig.pan} /></div></div>
                        </div>
                        {activeTool === 'ild' && ildResult && (
                            <div className="h-[35vh] md:h-[40vh] shrink-0 border-t border-slate-200 bg-white z-20 relative shadow-[0_-4px_10px_rgba(0,0,0,0.05)] transition-all duration-300 ease-in-out">
                                <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-blue-500 via-indigo-500 to-purple-500"></div>
                                <div className="p-4 md:p-6 h-full flex flex-col">
                                    <h3 className="text-xs md:text-sm font-bold text-slate-800 mb-2 uppercase tracking-wider flex items-center gap-2"><TrendingUp size={14} className="text-purple-600" />ILD for {ildTarget.type} @ {ildTarget.location}m</h3>
                                    <div className="flex-1 min-h-0">
                                        <ILDChart ildResult={ildResult} />
                                    </div>
                                </div>
                            </div>
                        )}
                    </div>
                )}
            </div>
        </div>
    );
}
