import { BeamConfig, AnalysisResult } from './types';

// CIVILBEAMCALCULATOR FEM ENGINE v2.8 (Sign Convention Fix)

export const TOLERANCE = 1e-9;
// Plotting step size in meters (1cm resolution)
const PLOT_STEP = 0.01; 

/**
 * Main Analysis Entry Point
 */
export function analyzeBeam(inputConfig: BeamConfig): AnalysisResult {
  try {
    // 1. Adapter: Convert flat UI config to Engine nested structure if needed
    // Clone to avoid mutations
    const config = JSON.parse(JSON.stringify(inputConfig));
    
    // Ensure the config has the structure the engine logic expects
    if (!config.spans) {
        config.spans = [{
            id: 'main-span',
            length: config.length,
            supports: config.supports,
            loads: [], // We put everything in globalLoads for the engine to handle
            hinges: config.hinges
        }];
    }
    if (!config.globalLoads) {
        config.globalLoads = config.loads;
    }

    // 1. Pre-process
    const normConfig = normalizeConfig(config);

    // 1b. Identify Hinges
    const hingeSet = new Set<number>();
    normConfig.spans.forEach((s: any) => s.hinges?.forEach((h: any) => {
        if(isValidNumber(h.x)) hingeSet.add(h.x);
    }));

    // 2. Mesh Generation
    const { meshNodes, nodeMap } = generateSmartMesh(normConfig);
    (normConfig as any)._meshNodes = meshNodes;
    (normConfig as any)._nodeMap = nodeMap;

    // 3. Assemble
    const { K, dofMap } = assembleGlobalStiffness(normConfig);
    const { F } = assembleLoadVector(normConfig, dofMap);

    // 4. Boundary Conditions
    const { K_red, F_red, map_free, restrainedDOFs, prescribedDisp } = applyBoundaryConditions(K, F, normConfig, dofMap);

    // 5. Solve
    let U_global: number[] = [];
    try {
        const U_red = solveLinearSystem(K_red, F_red);
        U_global = expandDisplacements(U_red, map_free, K.length, prescribedDisp);
    } catch (e) {
        // This is an expected "Unstable" error from the solver
        throw new Error("Structure is unstable. Please check supports.");
    }

    // 6. Post-Process
    return postProcessResults(normConfig, U_global, dofMap, restrainedDOFs, F, K);

  } catch (e) {
      // Catch-all for any engine crash (e.g. invalid inputs, mesh errors)
      console.warn("FEM Engine Exception:", e);
      // Return empty results on failure instead of crashing the app
      return {
          diagrams: { x: [], shear: [], moment: [], deflection: [], bendingStress: [] },
          reactions: {},
          summary: { maxShear: {x:0, value:0}, maxMoment: {x:0, value:0}, maxDeflection: {x:0, value:0}, maxStress: {x:0, value:0} }
      };
  }
}

// --- Implementation Details ---

function isValidNumber(n: any): boolean {
    return typeof n === 'number' && !isNaN(n) && isFinite(n);
}

function normalizeConfig(config: any) {
  const resolveLoad = (load: any) => {
      // Corrected Logic: Only apply angle resolution if explicitly provided.
      // Removed aggressive default angle=90 which was flipping signs for vertical loads.
      
      if (load.angle !== undefined && load.angle_deg === undefined) {
        load.angle_deg = load.angle;
      }
      
      if (load.angle_deg !== undefined) {
        const rad = (load.angle_deg) * Math.PI / 180;
        const mag = load.value || 0;
        load.fy = mag * Math.sin(rad); 
        // Do not negate fy. FEM Solver expects forces in Global Y (Up positive).
        // If user enters -10 (Down) and angle 90, sin(90)=1, fy=-10. 
        // Value passed to solver should be -10.
        load.value = load.fy; 
      } else {
        load.fy = load.value || 0;
      }

      if (load.type === 'udl' || load.type === 'uvl') {
          if (load.start === undefined) load.start = 0;
          if (load.end === undefined) load.end = config.length || 10;
      }
  };

  config.spans.forEach((span: any) => {
    if(span.loads) span.loads.forEach(resolveLoad);
  });
  if (config.globalLoads) config.globalLoads.forEach(resolveLoad);
  return config;
}

function generateSmartMesh(config: any) {
  const criticalPoints = new Set([0, config.length]);
  const hingeSet = new Set<number>();
  
  config.spans.forEach((s: any) => s.hinges?.forEach((h: any) => {
      if(isValidNumber(h.x)) hingeSet.add(h.x);
  }));

  const addLoadPoints = (loads: any[]) => {
    loads?.forEach(l => {
      if (l.type === 'point' || l.type === 'moment') {
        if(isValidNumber(l.x)) criticalPoints.add(l.x);
      } else if (l.type === 'udl' || l.type === 'uvl') {
        if(isValidNumber(l.start)) criticalPoints.add(l.start);
        if(isValidNumber(l.end)) criticalPoints.add(l.end);
      }
    });
  };

  config.spans.forEach((span: any) => {
    span.supports?.forEach((s: any) => {
        if(isValidNumber(s.x)) criticalPoints.add(s.x);
    });
    addLoadPoints(span.loads);
    span.hinges?.forEach((h: any) => {
        if(isValidNumber(h.x)) criticalPoints.add(h.x);
    });
  });
  
  addLoadPoints(config.globalLoads);
  
  config.sections?.forEach((sec: any) => {
      if(isValidNumber(sec.x)) criticalPoints.add(sec.x);
      if (isValidNumber(sec.end)) criticalPoints.add(sec.end);
  });

  // Filter invalid points
  let uniqueCoords = Array.from(criticalPoints)
      .filter(x => isValidNumber(x))
      .sort((a, b) => a - b);
  
  // Clean coordinates to prevent micro-gaps (snap to 6 decimals)
  uniqueCoords = uniqueCoords.map(x => Math.round(x * 1e6) / 1e6);
      
  uniqueCoords = uniqueCoords.filter((val, index, arr) => index === 0 || (val - arr[index - 1] > TOLERANCE));

  const MAX_EL_SIZE = config.targetElementSize || 0.5;
  const finalMesh: number[] = [];
  const nodeMap: Record<string, number> = {};

  if(uniqueCoords.length > 0) {
      finalMesh.push(uniqueCoords[0]);
      for (let i = 0; i < uniqueCoords.length - 1; i++) {
        const x0 = uniqueCoords[i];
        const x1 = uniqueCoords[i + 1];
        const L = x1 - x0;

        if (L > MAX_EL_SIZE + TOLERANCE) {
          const nSub = Math.ceil(L / MAX_EL_SIZE);
          for (let k = 1; k < nSub; k++) {
            // Using integer math here for intermediate points isn't strictly necessary 
            // as long as we snap, but standard division is okay for subdivision
            finalMesh.push(x0 + (L * k) / nSub);
          }
        }
        finalMesh.push(x1);
        if (hingeSet.has(x1) && x1 > TOLERANCE && x1 < config.length - TOLERANCE) {
          finalMesh.push(x1);
        }
      }
  }

  const meshNodes = finalMesh.sort((a, b) => a - b);
  meshNodes.forEach((x, i) => {
    // Guard against null/NaN keys
    if(isValidNumber(x)) {
        const key = x.toFixed(9);
        if (nodeMap[key] === undefined) nodeMap[key] = i;
    }
  });
  
  return { meshNodes, nodeMap };
}

function assembleGlobalStiffness(config: any) {
  const nodes = config._meshNodes;
  const nNodes = nodes.length;
  const dofMap: number[][] = [];
  let dofCounter = 0;

  for (let i = 0; i < nNodes; i++) {
    const isSplitNode = (i > 0 && Math.abs(nodes[i] - nodes[i-1]) < TOLERANCE);
    if (isSplitNode) {
      const vDof = dofMap[i-1][0];
      const tDof = dofCounter++;
      dofMap.push([vDof, tDof]);
    } else {
      const vDof = dofCounter++;
      const tDof = dofCounter++;
      dofMap.push([vDof, tDof]);
    }
  }

  const nDOF = dofCounter;
  const K = Array.from({length: nDOF}, () => Array(nDOF).fill(0));
  
  for (let i = 0; i < nNodes - 1; i++) {
    const L = nodes[i+1] - nodes[i];
    if (L < TOLERANCE) continue;

    const mid = (nodes[i] + nodes[i+1]) / 2;
    const {E, I} = getSectionProps(config, mid); 
    const k_el = beamElementStiffness(E, I, L);
    const indexes = [...dofMap[i], ...dofMap[i+1]];
    
    for (let r = 0; r < 4; r++) {
      for (let c = 0; c < 4; c++) {
        K[indexes[r]][indexes[c]] += k_el[r][c];
      }
    }
  }
  return { K, dofMap };
}

function assembleLoadVector(config: any, dofMap: number[][]) {
  const nodes = config._meshNodes;
  let maxDof = 0;
  dofMap.forEach(d => { maxDof = Math.max(maxDof, d[0], d[1]); });
  const nDOF = maxDof + 1;
  const F = Array(nDOF).fill(0);

  for (let i = 0; i < nodes.length - 1; i++) {
    const xStart = nodes[i];
    const xEnd = nodes[i+1];
    const L = xEnd - xStart;
    if (L < TOLERANCE) continue;
    
    const processLoads = (loads: any[]) => {
      if (!loads) return;
      loads.forEach(load => {
        if (load.type === 'point' || load.type === 'moment') {
          if (isValidNumber(load.x) && Math.abs(load.x - xStart) < TOLERANCE) {
             if (load.type === 'point') F[dofMap[i][0]] += (load.value || 0);
             if (load.type === 'moment') F[dofMap[i][1]] += (load.value || 0);
          }
        } else if (load.type === 'udl' || load.type === 'uvl') {
          const start = Math.max(load.start || 0, xStart);
          const end = Math.min(load.end || 0, xEnd);
          if (end > start + TOLERANCE) {
             let w1, w2;
             if (load.type === 'udl') {
               w1 = w2 = (load.value || 0);
             } else {
               const vStart = load.valueStart || 0;
               const vEnd = load.valueEnd || 0;
               const lStart = load.start || 0;
               const lEnd = load.end || 0;
               const m = (vEnd - vStart) / (lEnd - lStart);
               w1 = vStart + m * (xStart - lStart);
               w2 = vStart + m * (xEnd - lStart);
             }
             const fem = calculateFixedEndActions(w1, w2, L);
             F[dofMap[i][0]]   += fem[0];
             F[dofMap[i][1]]   += fem[1];
             F[dofMap[i+1][0]] += fem[2];
             F[dofMap[i+1][1]] += fem[3];
          }
        }
      });
    };
    config.spans.forEach((s: any) => processLoads(s.loads));
    processLoads(config.globalLoads);
  }

  // Last node point loads
  if(nodes.length > 0) {
      const lastNodeIdx = nodes.length - 1;
      const xLast = nodes[lastNodeIdx];
      const processLastNodeLoads = (loads: any[]) => {
        if (!loads) return;
        loads.forEach(load => {
          if ((load.type === 'point' || load.type === 'moment') && isValidNumber(load.x) && Math.abs(load.x - xLast) < TOLERANCE) {
             if (load.type === 'point') F[dofMap[lastNodeIdx][0]] += (load.value || 0);
             if (load.type === 'moment') F[dofMap[lastNodeIdx][1]] += (load.value || 0);
          }
        });
      };
      config.spans.forEach((s: any) => processLastNodeLoads(s.loads));
      processLastNodeLoads(config.globalLoads);
  }

  return { F };
}

function calculateFixedEndActions(w1: number, w2: number, L: number) {
  // Calculates Equivalent Nodal Loads (actions on nodes) 
  // for a fixed-fixed element with trapezoidal load.
  // Formulas derived from standard Fixed End Moments but signed for Nodal Forces.
  // For Downward load (w < 0):
  // Fy should be Down (Negative).
  // M_left should be CW (Negative).
  // M_right should be CCW (Positive).
  const fy1 = (L/20) * (7*w1 + 3*w2);
  const m1  = (L*L/60) * (3*w1 + 2*w2);
  const fy2 = (L/20) * (3*w1 + 7*w2);
  const m2  = -(L*L/60) * (2*w1 + 3*w2);
  return [fy1, m1, fy2, m2];
}

function applyBoundaryConditions(K: number[][], F: number[], config: any, dofMap: number[][]) {
  const restrainedDOFs = new Set<number>();
  const prescribedDisp: Record<number, number> = {};
  
  // Safe node map lookup
  const getNodeIndex = (x: number) => {
      if(!isValidNumber(x)) return undefined;
      return config._nodeMap[x.toFixed(9)];
  }

  config.spans.forEach((span: any) => {
    span.supports?.forEach((s: any) => {
      const n = getNodeIndex(s.x);
      if (n === undefined) return;
      
      if (s.type === 'pin' || s.type === 'roller') {
        restrainedDOFs.add(dofMap[n][0]);
        if (s.settlement) prescribedDisp[dofMap[n][0]] = s.settlement;
      } else if (s.type === 'fixed') {
        restrainedDOFs.add(dofMap[n][0]);
        restrainedDOFs.add(dofMap[n][1]);
        if (s.settlement) prescribedDisp[dofMap[n][0]] = s.settlement;
        if (s.rotation) prescribedDisp[dofMap[n][1]] = s.rotation;
      } else if (s.type === 'spring_y') {
         K[dofMap[n][0]][dofMap[n][0]] += (s.stiffness || 0);
      }
    });
  });

  const nDOF = K.length;
  const map_free: number[] = [];
  for(let i=0; i<nDOF; i++) {
    if(!restrainedDOFs.has(i)) map_free.push(i);
  }

  const K_red: number[][] = [];
  const F_red: number[] = [];
  
  for(let i of map_free) {
    const row: number[] = [];
    let f_val = F[i];
    for(let j of map_free) {
      row.push(K[i][j]);
    }
    for(let k in prescribedDisp) {
      const idx = parseInt(k);
      f_val -= K[i][idx] * prescribedDisp[idx];
    }
    K_red.push(row);
    F_red.push(f_val);
  }

  return { K_red, F_red, map_free, restrainedDOFs, prescribedDisp };
}

function solveLinearSystem(A: number[][], b: number[]) {
  const n = A.length;
  if(n === 0) throw new Error("System is empty");

  const M = A.map(row => [...row]);
  const x = [...b];
  
  for (let i = 0; i < n; i++) {
    let maxRow = i;
    for (let k = i + 1; k < n; k++) {
      if (Math.abs(M[k][i]) > Math.abs(M[maxRow][i])) maxRow = k;
    }
    if (Math.abs(M[maxRow][i]) < TOLERANCE) {
        throw new Error("Singular Matrix");
    }
    [M[i], M[maxRow]] = [M[maxRow], M[i]];
    [x[i], x[maxRow]] = [x[maxRow], x[i]];
    for (let k = i + 1; k < n; k++) {
      const factor = M[k][i] / M[i][i];
      x[k] -= factor * x[i];
      for (let j = i; j < n; j++) {
        M[k][j] -= factor * M[i][j];
      }
    }
  }
  for (let i = n - 1; i >= 0; i--) {
    let sum = 0;
    for (let j = i + 1; j < n; j++) sum += M[i][j] * x[j];
    x[i] = (x[i] - sum) / M[i][i];
  }
  return x;
}

function expandDisplacements(u_red: number[], map_free: number[], nDOF: number, prescribedDisp: any) {
  const u = Array(nDOF).fill(0);
  map_free.forEach((globalIdx, k) => {
    u[globalIdx] = u_red[k];
  });
  if (prescribedDisp) {
    for (const [dof, val] of Object.entries(prescribedDisp)) {
      u[parseInt(dof)] = (val as number);
    }
  }
  return u;
}

function postProcessResults(config: any, U: number[], dofMap: number[][], restrainedDOFs: any, F_global: number[], K: number[][]) {
  const nodes = config._meshNodes;
  const diagrams = { x: [] as number[], shear: [] as number[], moment: [] as number[], deflection: [] as number[], bendingStress: [] as number[] };
  const reactions: Record<number, any> = {};

  for (let i = 0; i < nodes.length - 1; i++) {
    const xStart = nodes[i];
    const xEnd = nodes[i+1];
    const L = xEnd - xStart;
    if (L < TOLERANCE) continue;
    
    const idx1 = dofMap[i];
    const idx2 = dofMap[i+1];
    let u_elem = [ U[idx1[0]], U[idx1[1]], U[idx2[0]], U[idx2[1]] ];
    
    const mid = (xStart + xEnd) / 2;
    const {E, I, depth} = getSectionProps(config, mid);
    const k_el = beamElementStiffness(E, I, L);
    const f_nodal = multiplyMatrixVector(k_el, u_elem);
    
    const loadsOnElem = getLoadsOnElement(config, xStart, xEnd);
    const fea = calculateTotalFEA(loadsOnElem, xStart, xEnd, L); 
    
    const V_start_val = f_nodal[0] - fea[0];
    const M_start_val = f_nodal[1] - fea[1];
    
    let V_current = V_start_val;
    let M_current = -M_start_val; 

    // Robust grid generation: Use integer loop to avoid floating point misses (e.g. 5.0 vs 4.999)
    const samplePoints: number[] = [xStart];
    
    // We want to cover every 0.01m global mark that falls strictly within (xStart, xEnd)
    // Convert to integers (cm) for robust comparison
    const startCm = Math.floor(xStart * 100);
    const endCm = Math.ceil(xEnd * 100);
    
    for (let cm = startCm + 1; cm < endCm; cm++) {
        const x = cm / 100;
        // Double check it's strictly inside to avoid duplicating nodes
        if (x > xStart + 1e-6 && x < xEnd - 1e-6) {
            samplePoints.push(x);
        }
    }
    samplePoints.push(xEnd);

    for (let k = 0; k < samplePoints.length; k++) {
       const x = samplePoints[k];
       const xi = (x - xStart) / L; // Local coordinate
       
       const y = hermiteInterpolation(u_elem, xi, L);
       
       let loadShear = 0;
       let loadMoment = 0;
       
       loadsOnElem.forEach(l => {
         if (l.type === 'udl' || l.type === 'uvl') {
           const segLen = x - xStart;
           if(segLen > 0) {
               const w_a = getLoadIntensity(l, xStart);
               const w_x = getLoadIntensity(l, x);
               const P = (w_a + w_x) / 2 * segLen;
               const segmentMoment = (segLen * segLen / 6) * (2 * w_a + w_x);

               loadShear += P; 
               loadMoment += segmentMoment;
           }
         }
       });
       
       const V_x = V_current + loadShear;
       const M_x = M_current + V_current * (x - xStart) + loadMoment;
       
       // Stress Calculation: sigma = M * y / I
       // M is kNm, y is m, I is m^4 => result is kPa
       // We use depth/2 as y_max for symmetric sections
       const y_max = (depth || 0) / 2;
       const sigma_kPa = I > 0 ? (Math.abs(M_x) * y_max / I) : 0;

       // Handle discontinuity (vertical line) at start of element if needed
       if (k === 0 && diagrams.x.length > 0) {
         const lastX = diagrams.x[diagrams.x.length - 1];
         // Only if we are truly at the same x (continuity)
         if (Math.abs(lastX - x) < 1e-9) {
             diagrams.x.push(x);
             diagrams.shear.push(V_x);
             diagrams.moment.push(M_x);
             diagrams.deflection.push(y);
             diagrams.bendingStress.push(sigma_kPa);
         } else {
             diagrams.x.push(x);
             diagrams.shear.push(V_x);
             diagrams.moment.push(M_x);
             diagrams.deflection.push(y);
             diagrams.bendingStress.push(sigma_kPa);
         }
       } else if (k > 0 || diagrams.x.length === 0) {
           diagrams.x.push(x);
           diagrams.shear.push(V_x);
           diagrams.moment.push(M_x);
           diagrams.deflection.push(y);
           diagrams.bendingStress.push(sigma_kPa);
       }
    }
  }

  // Precision Rounding Step
  diagrams.x = diagrams.x.map(val => Math.round(val * 10000) / 10000);
  
  const R = new Float64Array(F_global.length);
  for (let i = 0; i < F_global.length; i++) {
    let f_int = 0;
    for (let j = 0; j < U.length; j++) f_int += K[i][j] * U[j];
    R[i] = f_int - F_global[i];
  }

  config.spans.forEach((span: any) => {
    span.supports?.forEach((s: any) => {
      // Guard node lookup
      if(isValidNumber(s.x)) {
          const n = config._nodeMap[s.x.toFixed(9)];
          if (n !== undefined) {
            let fy = R[dofMap[n][0]];
            let mz = R[dofMap[n][1]];
            if (s.type === 'spring_y') fy = -s.stiffness * U[dofMap[n][0]];
            reactions[s.x] = { Fy: fy, Mz: mz };
          }
      }
    });
  });

  const summary = {
    maxShear: { value: 0, x: 0 },
    maxMoment: { value: 0, x: 0 },
    maxDeflection: { value: 0, x: 0 },
    maxStress: { value: 0, x: 0 }
  };

  const findAbsMax = (arr: number[]) => {
    let maxV = 0;
    let maxX = 0;
    for (let i = 0; i < arr.length; i++) {
      if (Math.abs(arr[i]) > Math.abs(maxV)) {
        maxV = arr[i];
        maxX = diagrams.x[i];
      }
    }
    return { value: maxV, x: maxX };
  };

  if (diagrams.x.length > 0) {
    summary.maxShear = findAbsMax(diagrams.shear);
    summary.maxMoment = findAbsMax(diagrams.moment);
    summary.maxDeflection = findAbsMax(diagrams.deflection);
    summary.maxStress = findAbsMax(diagrams.bendingStress);
  }

  return { diagrams, reactions, summary };
}

function getLoadIntensity(load: any, x: number) {
  if (load.type === 'udl') return (load.value || 0);
  if (load.type === 'uvl') {
    if (x < (load.start || 0) - TOLERANCE || x > (load.end || 0) + TOLERANCE) return 0;
    const lStart = load.start || 0;
    const lEnd = load.end || 0;
    const effX = Math.max(lStart, Math.min(x, lEnd));
    const ratio = (effX - lStart) / (lEnd - lStart);
    return (load.valueStart || 0) + ratio * ((load.valueEnd || 0) - (load.valueStart || 0));
  }
  return 0;
}

function getLoadsOnElement(config: any, xStart: number, xEnd: number) {
  const active: any[] = [];
  const check = (loads: any[]) => {
    if (!loads) return;
    loads.forEach(l => {
      if (l.type === 'udl' || l.type === 'uvl') {
         if (Math.min(l.end || 0, xEnd) - Math.max(l.start || 0, xStart) > TOLERANCE) {
           active.push(l);
         }
      }
    });
  };
  config.spans.forEach((s: any) => check(s.loads));
  check(config.globalLoads);
  return active;
}

function calculateTotalFEA(loads: any[], xStart: number, xEnd: number, L: number) {
  let f = [0,0,0,0];
  loads.forEach(l => {
     let w1 = getLoadIntensity(l, xStart);
     let w2 = getLoadIntensity(l, xEnd);
     let res = calculateFixedEndActions(w1, w2, L);
     for(let i=0; i<4; i++) f[i] += res[i];
  });
  return f;
}

function hermiteInterpolation(u: number[], xi: number, L: number) {
  const [v1, t1, v2, t2] = u;
  const xi2 = xi*xi;
  const xi3 = xi*xi*xi;
  
  const N1 = 1 - 3*xi2 + 2*xi3;
  const N2 = L * (xi - 2*xi2 + xi3);
  const N3 = 3*xi2 - 2*xi3;
  const N4 = L * (-xi2 + xi3);
  
  return N1*v1 + N2*t1 + N3*v2 + N4*t2;
}

function getSectionProps(config: any, x: number) {
  const defaults = { E: config.E || 2e8, I: config.I || 1e-4, depth: config.depth || 0.5 };
  if (!config.sections || config.sections.length === 0) {
    return defaults;
  }
  // Check range for each section
  for (const sec of config.sections) {
    const end = sec.end !== undefined ? sec.end : config.length;
    if (x >= sec.x - TOLERANCE && x <= end + TOLERANCE) {
        return { 
            E: sec.E, 
            I: sec.I,
            depth: sec.depth !== undefined ? sec.depth : defaults.depth
        };
    }
  }
  return defaults;
}

function beamElementStiffness(E: number, I: number, L: number) {
  const k = E * I / (L * L * L);
  return [
    [12*k,      6*L*k,     -12*k,     6*L*k],
    [6*L*k,     4*L*L*k,   -6*L*k,    2*L*L*k],
    [-12*k,    -6*L*k,      12*k,    -6*L*k],
    [6*L*k,     2*L*L*k,   -6*L*k,    4*L*L*k]
  ];
}

function multiplyMatrixVector(M: number[][], v: number[]) {
  return M.map(row => row.reduce((sum, val, i) => sum + val * v[i], 0));
}
