import React from 'react';
import { BeamConfig } from './types';

interface Props {
    config: BeamConfig;
    selectedILDPoint?: number | null;
    onSelectILDPoint?: (x: number) => void;
    zoom?: number;
    pan?: { x: number; y: number };
}

const LoadLabel = ({ x, y, text, color = "#475569" }: { x: number, y: number, text: string, color?: string }) => {
    // Estimate width: ~7px per char for monospace 11px + padding
    const width = Math.max(40, text.length * 7 + 12);
    const height = 20;

    return (
        <g transform={`translate(${x}, ${y})`} className="select-none pointer-events-none">
            <defs>
                <filter id="label-shadow" x="-50%" y="-50%" width="200%" height="200%">
                    <feDropShadow dx="0" dy="1" stdDeviation="2" floodColor="#000000" floodOpacity="0.08"/>
                </filter>
            </defs>
            <rect 
                x={-width/2} 
                y={-height/2} 
                width={width} 
                height={height} 
                rx="4" 
                fill="white" 
                stroke="#cbd5e1" 
                strokeWidth="1"
                filter="url(#label-shadow)"
            />
            <text 
                x="0" 
                y="4" 
                textAnchor="middle" 
                fontSize="11" 
                fill={color} 
                fontWeight="600" 
                fontFamily="JetBrains Mono, monospace"
            >
                {text}
            </text>
        </g>
    );
};

const BeamVisualizer: React.FC<Props> = ({ config, selectedILDPoint, onSelectILDPoint, zoom = 1, pan = { x: 0, y: 0 } }) => {
    const { length, supports, loads, hinges, sections } = config;

    // Scaling constants
    const padding = 60;
    const height = 360; 
    const width = 800; // viewBox width
    
    // Scale X to fit in width
    const scaleX = (x: number) => padding + (x / length) * (width - 2 * padding);
    const beamY = 180; // Beam Axis Y centered vertically

    // BEAM PROFILE GENERATION
    // 1. Collect all critical X points for geometry changes
    const getProfileSegments = () => {
        const points = new Set<number>([0, length]);
        sections?.forEach(s => {
            if (s.x !== undefined) points.add(Math.max(0, Math.min(length, s.x)));
            if (s.end !== undefined) points.add(Math.max(0, Math.min(length, s.end)));
        });
        
        const sortedX = Array.from(points).sort((a,b) => a - b);
        const segments = [];

        for (let i = 0; i < sortedX.length - 1; i++) {
            const xStart = sortedX[i];
            const xEnd = sortedX[i+1];
            if (xEnd - xStart < 0.001) continue;
            
            const mid = (xStart + xEnd) / 2;
            
            // Find active depth for this segment
            let activeDepth = config.depth || 0.5;
            
            // Priority: First section in the list that covers this point wins (matching engine logic)
            if (sections) {
                for (const sec of sections) {
                    const sEnd = sec.end !== undefined ? sec.end : length;
                    if (mid >= sec.x && mid <= sEnd) {
                        activeDepth = sec.depth || activeDepth;
                        break;
                    }
                }
            }
            
            segments.push({ xStart, xEnd, depth: activeDepth });
        }
        return segments;
    };

    const beamSegments = getProfileSegments();
    const maxRenderDepth = 1.2; // Limit max visual thickness
    const depthScalePx = 40; // 1m depth = 40px

    // 1. Calculate Dimension Points
    const getDimensionPoints = () => {
        const points = new Set<number>([0, length]);
        supports.forEach(s => points.add(s.x));
        loads.forEach(l => {
            if (l.type === 'point' || l.type === 'moment') points.add(l.x ?? 0);
            if (l.type === 'udl' || l.type === 'uvl') {
                points.add(l.start ?? 0);
                points.add(l.end ?? length);
            }
        });
        hinges.forEach(h => points.add(h.x));
        
        return Array.from(points)
            .sort((a, b) => a - b)
            .filter((v, i, a) => i === 0 || v - a[i-1] > 0.05); 
    };

    const dimPoints = getDimensionPoints();
    
    // Dimension Y positions
    const dimYChain = beamY + 80;
    const dimYTotal = beamY + 110;

    // Helper: Determine max load magnitude for scaling
    let maxLoadMag = 1;
    loads.forEach(l => {
        if (l.type === 'point') maxLoadMag = Math.max(maxLoadMag, Math.abs(l.value));
        if (l.type === 'udl') maxLoadMag = Math.max(maxLoadMag, Math.abs(l.value));
        if (l.type === 'uvl') maxLoadMag = Math.max(maxLoadMag, Math.abs(l.valueStart||0), Math.abs(l.valueEnd||0));
    });
    if (maxLoadMag === 0) maxLoadMag = 10; 
    
    const MAX_PIXEL_HEIGHT = 70; // Slightly taller max height
    const MIN_PIXEL_HEIGHT = 30;
    
    const getLoadHeight = (val: number) => {
        if (val === 0) return 0;
        const abs = Math.abs(val);
        const ratio = abs / maxLoadMag;
        return Math.max(MIN_PIXEL_HEIGHT, ratio * MAX_PIXEL_HEIGHT);
    };

    // Render Helpers
    const renderDistributedLoad = (l: any) => {
        const startX = scaleX(l.start || 0);
        const endX = scaleX(l.end || 0);
        const wPx = endX - startX;
        
        if (wPx <= 0) return null;

        const vStart = l.type === 'udl' ? l.value : (l.valueStart || 0);
        const vEnd = l.type === 'udl' ? l.value : (l.valueEnd || 0);
        
        const isDown = (vStart + vEnd) / 2 <= 0; 
        
        const hStart = getLoadHeight(vStart);
        const hEnd = getLoadHeight(vEnd);
        
        // Load sits on top surface of beam if possible, but for simple schematic we typically use axis
        // To be cleaner, let's keep it relative to axis `beamY` to avoid jumping around with variable depth
        const yBase = beamY; 

        // If Down: tails are above beam (y < beamY)
        const yTailStart = isDown ? yBase - hStart : yBase + hStart;
        const yTailEnd = isDown ? yBase - hEnd : yBase + hEnd;
        
        // Arrow spacing
        const spacing = 12;
        const numArrows = Math.ceil(wPx / spacing);
        const arrowEls = [];
        
        for (let i = 0; i <= numArrows; i++) {
            const ratio = i / numArrows;
            const x = startX + ratio * wPx;
            const yTail = yTailStart + ratio * (yTailEnd - yTailStart);
            
            arrowEls.push(
                <line 
                    key={`arr-${i}`} 
                    x1={x} y1={yTail} 
                    x2={x} y2={yBase} 
                    stroke="#0ea5e9" 
                    strokeWidth="1"
                    markerEnd="url(#arrowhead-blue)" 
                    opacity="0.6"
                />
            );
        }

        // Connecting Line (The "Comb" back)
        arrowEls.push(
            <line 
                key="comb-back"
                x1={startX} y1={yTailStart}
                x2={endX} y2={yTailEnd}
                stroke="#0ea5e9"
                strokeWidth="2"
            />
        );

        // Label Positioning
        const labelVal = l.type === 'udl' 
            ? `${Math.abs(vStart)} kN/m` 
            : `${Math.abs(vStart)} → ${Math.abs(vEnd)} kN/m`;
        
        const minY = Math.min(yTailStart, yTailEnd, yBase);
        const maxY = Math.max(yTailStart, yTailEnd, yBase);
        
        const labelY = isDown ? minY - 15 : maxY + 15;
        
        return (
            <g key={l.id}>
                <defs>
                    <linearGradient id={`grad-${l.id}`} x1="0%" y1="0%" x2="0%" y2="100%">
                        <stop offset="0%" stopColor="#bae6fd" stopOpacity="0.3" />
                        <stop offset="100%" stopColor="#bae6fd" stopOpacity="0.05" />
                    </linearGradient>
                </defs>
                <path 
                    d={`M${startX},${yBase} L${startX},${yTailStart} L${endX},${yTailEnd} L${endX},${yBase} Z`} 
                    fill={`url(#grad-${l.id})`} 
                />
                {arrowEls}
                <LoadLabel 
                    x={(startX + endX)/2} 
                    y={labelY} 
                    text={labelVal} 
                    color="#0369a1" 
                />
            </g>
        );
    };

    const handleSvgClick = (e: React.MouseEvent<SVGSVGElement>) => {
        if (!onSelectILDPoint) return;
        const rect = e.currentTarget.getBoundingClientRect();
        // Adjust click coordinate to account for zoom/pan
        // Center of zoom is width/2, height/2
        const cx = width / 2;
        const cy = height / 2;
        
        const rawX = e.clientX - rect.left;
        const rawY = e.clientY - rect.top;
        
        // Map raw px to viewBox px
        const vx = rawX * (width / rect.width);
        const vy = rawY * (height / rect.height);
        
        // Reverse transform: translate(-cx-pan) scale(1/zoom) translate(cx)
        const tx = (vx - cx - pan.x) / zoom + cx;
        
        let x_m = ((tx - padding) / (width - 2 * padding)) * length;
        x_m = Math.max(0, Math.min(length, x_m));
        x_m = Math.round(x_m * 10) / 10;
        onSelectILDPoint(x_m);
    };

    return (
        <div className="w-full overflow-hidden bg-white rounded-xl shadow-sm border border-slate-200 group relative">
            <svg 
                viewBox={`0 0 ${width} ${height}`} 
                className={`w-full h-auto ${onSelectILDPoint ? 'cursor-crosshair' : ''}`}
                onClick={handleSvgClick}
            >
                <defs>
                    <pattern id="dotGrid" width="20" height="20" patternUnits="userSpaceOnUse">
                         <circle cx="1" cy="1" r="1.5" fill="#f1f5f9" />
                    </pattern>
                    <marker id="arrowhead-red" markerWidth="6" markerHeight="4" refX="5" refY="2" orient="auto">
                        <path d="M0,0 L6,2 L0,4 L0,0" fill="#f43f5e" />
                    </marker>
                    <marker id="arrowhead-blue" markerWidth="6" markerHeight="4" refX="5" refY="2" orient="auto">
                        <path d="M0,0 L6,2 L0,4 L0,0" fill="#0ea5e9" />
                    </marker>
                    <marker id="arrowhead-moment" markerWidth="6" markerHeight="6" refX="5" refY="3" orient="auto">
                         <path d="M0,0 L6,3 L0,6 L0,0" fill="#f43f5e" />
                    </marker>
                    <marker id="dim-arrow-start" markerWidth="6" markerHeight="6" refX="0" refY="3" orient="auto">
                        <path d="M6,0 L0,3 L6,6" fill="#94a3b8" />
                    </marker>
                    <marker id="dim-arrow-end" markerWidth="6" markerHeight="6" refX="6" refY="3" orient="auto">
                        <path d="M0,0 L6,3 L0,6" fill="#94a3b8" />
                    </marker>
                </defs>
                <rect width="100%" height="100%" fill="url(#dotGrid)" />

                <g transform={`translate(${width/2 + pan.x}, ${height/2 + pan.y}) scale(${zoom}) translate(${-width/2}, ${-height/2})`}>
                    {/* Variable Depth Beam Segments */}
                    <g>
                        {beamSegments.map((seg, i) => {
                            const sx = scaleX(seg.xStart);
                            const ex = scaleX(seg.xEnd);
                            // clamp visual thickness
                            const thickness = Math.min(seg.depth, maxRenderDepth) * depthScalePx; 
                            
                            return (
                                <g key={`beam-seg-${i}`}>
                                    <rect 
                                        x={sx} 
                                        y={beamY - thickness/2} 
                                        width={ex - sx} 
                                        height={thickness} 
                                        fill="#1e293b" 
                                        stroke="#0f172a" 
                                        strokeWidth="1"
                                    />
                                    {/* Centerline reference */}
                                    <line 
                                        x1={sx} y1={beamY} 
                                        x2={ex} y2={beamY} 
                                        stroke="#334155" 
                                        strokeWidth="1" 
                                        strokeDasharray="4 4" 
                                        opacity="0.5" 
                                    />
                                </g>
                            )
                        })}
                    </g>

                    {/* Supports */}
                    {supports.map(s => {
                        const sx = scaleX(s.x);
                        const size = 14;
                        // Find depth at this support location to position it correctly at bottom of beam
                        let localDepth = config.depth || 0.5;
                        if (sections) {
                            for (const sec of sections) {
                                const end = sec.end !== undefined ? sec.end : length;
                                if (s.x >= sec.x && s.x <= end) {
                                    localDepth = sec.depth || localDepth;
                                    break;
                                }
                            }
                        }
                        const halfThickness = (Math.min(localDepth, maxRenderDepth) * depthScalePx) / 2;
                        const supportY = beamY + halfThickness;
                        
                        return (
                            <g key={s.id} transform={`translate(${sx}, ${supportY})`}>
                                {s.type === 'pin' && (
                                    <g>
                                        <path d={`M0,0 L-${size/2},${size} L${size/2},${size} Z`} fill="#f1f5f9" stroke="#334155" strokeWidth="2" strokeLinejoin="round" />
                                        <line x1={-size} y1={size} x2={size} y2={size} stroke="#334155" strokeWidth="2" strokeLinecap="round" />
                                        <g stroke="#94a3b8" strokeWidth="1.5">
                                            <line x1={-size} y1={size} x2={-size-5} y2={size+5} />
                                            <line x1={-size+5} y1={size} x2={-size} y2={size+5} />
                                            <line x1={-size+10} y1={size} x2={-size+5} y2={size+5} />
                                            <line x1={size-5} y1={size} x2={size-10} y2={size+5} />
                                            <line x1={size} y1={size} x2={size-5} y2={size+5} />
                                        </g>
                                    </g>
                                )}
                                {s.type === 'roller' && (
                                    <g>
                                        <circle cx="0" cy={size/2} r={size/2} fill="#f1f5f9" stroke="#334155" strokeWidth="2" />
                                        <line x1={-size} y1={size} x2={size} y2={size} stroke="#334155" strokeWidth="2" strokeLinecap="round" />
                                        <g stroke="#94a3b8" strokeWidth="1.5">
                                            <line x1={-size} y1={size} x2={-size-5} y2={size+5} />
                                            <line x1={-size+5} y1={size} x2={-size} y2={size+5} />
                                            <line x1={0} y1={size} x2={-5} y2={size+5} />
                                            <line x1={size} y1={size} x2={size-5} y2={size+5} />
                                        </g>
                                    </g>
                                )}
                                {s.type === 'fixed' && (
                                    <g>
                                        {/* Fixed support attaches to the side (usually left/right) but here we generalize to 'grounded' */}
                                        <line x1="0" y1={-halfThickness*2} x2="0" y2={size} stroke="#334155" strokeWidth="4" strokeLinecap="round" />
                                        {[...Array(6)].map((_, i) => (
                                            <line key={i} x1="0" y1={-halfThickness + i*5} x2={-8} y2={-halfThickness + i*5 + 8} stroke="#94a3b8" strokeWidth="1.5" />
                                        ))}
                                    </g>
                                )}
                                {s.type === 'spring_y' && (
                                    <g>
                                        <path d={`M0,0 l-5,6 l10,6 l-10,6 l10,6 l-5,6`} fill="none" stroke="#334155" strokeWidth="2" strokeLinejoin="round" />
                                        <line x1={-10} y1={30} x2={10} y2={30} stroke="#334155" strokeWidth="2" strokeLinecap="round" />
                                        <g stroke="#94a3b8" strokeWidth="1.5">
                                            <line x1={-8} y1={30} x2={-13} y2={35} />
                                            <line x1={0} y1={30} x2={-5} y2={35} />
                                            <line x1={8} y1={30} x2={3} y2={35} />
                                        </g>
                                    </g>
                                )}
                            </g>
                        );
                    })}

                    {/* Internal Hinges */}
                    {hinges.map(h => (
                        <g key={h.id} transform={`translate(${scaleX(h.x)}, ${beamY})`}>
                            <circle r="4" fill="white" stroke="#1e293b" strokeWidth="2" />
                        </g>
                    ))}

                    {/* Loads */}
                    {loads.map(l => {
                        const lx = scaleX(l.x !== undefined ? l.x : (l.start || 0));
                        const val = l.value || 0;
                        
                        if (l.type === 'udl' || l.type === 'uvl') {
                            return renderDistributedLoad(l);
                        }

                        if (l.type === 'point') {
                            const len = getLoadHeight(val);
                            const angleDeg = l.angle_deg !== undefined ? l.angle_deg : 90;
                            const rad = angleDeg * Math.PI / 180;
                            const s = Math.sign(val) === 0 ? 1 : Math.sign(val);
                            // SVG Y is down, so negative sin for visual up/down logic relative to math angles
                            const vecX = len * s * Math.cos(rad);
                            const vecY = len * s * Math.sin(rad) * -1; 
                            
                            const tailX = lx - vecX;
                            const tailY = beamY - vecY;

                            // Label placement logic: push away from tail along the vector line
                            // Normalized vector (-cos, sin) -> direction from head to tail
                            // We want to go further out from tail
                            const labelDist = 20;
                            const labelX = tailX - (vecX / len) * labelDist;
                            const labelY = tailY - (vecY / len) * labelDist;
                            
                            // Simple fallback for vertical loads to ensure clean alignment
                            const isVertical = Math.abs(Math.abs(angleDeg) - 90) < 1;
                            const finalLabelX = isVertical ? tailX : labelX;
                            const finalLabelY = isVertical ? (tailY < beamY ? tailY - 15 : tailY + 15) : labelY;

                            return (
                                <g key={l.id}>
                                    <line 
                                        x1={tailX} y1={tailY} 
                                        x2={lx} y2={beamY} 
                                        stroke="#f43f5e" 
                                        strokeWidth="2" 
                                        markerEnd="url(#arrowhead-red)" 
                                    />
                                    <LoadLabel 
                                        x={finalLabelX} 
                                        y={finalLabelY} 
                                        text={`${Math.abs(val)} kN`} 
                                        color="#b91c1c" 
                                    />
                                    {!isVertical && (
                                        <text x={tailX + 10} y={tailY} fontSize="9" fill="#94a3b8">{angleDeg}°</text>
                                    )}
                                </g>
                            );
                        }
                        
                        if (l.type === 'moment') {
                            const isCCW = val > 0;
                            const path = isCCW 
                                ? "M -15,-15 A 20,20 0 1,0 15,0" 
                                : "M 15,-15 A 20,20 0 1,1 -15,0"; 

                            return (
                                <g key={l.id} transform={`translate(${lx}, ${beamY})`}>
                                    <path d={path} fill="none" stroke="#f43f5e" strokeWidth="2" markerEnd="url(#arrowhead-moment)" />
                                    <LoadLabel 
                                        x={0} 
                                        y={-38} 
                                        text={`${Math.abs(val)} kNm`} 
                                        color="#b91c1c" 
                                    />
                                    <circle r="3" fill="#f43f5e" />
                                </g>
                            );
                        }
                        return null;
                    })}

                    {/* Dimensions */}
                    <g transform={`translate(0, 0)`} opacity="0.8">
                        {dimPoints.map((pt, i) => {
                            if (i === dimPoints.length - 1) return null;
                            const nextPt = dimPoints[i+1];
                            const x1 = scaleX(pt);
                            const x2 = scaleX(nextPt);
                            const dist = nextPt - pt;
                            if (x2 - x1 < 15) return null;

                            return (
                                <g key={`dim-${i}`}>
                                    <line x1={x1} y1={beamY + 25} x2={x1} y2={dimYChain + 5} stroke="#cbd5e1" strokeWidth="1" strokeDasharray="2 2"/>
                                    <line x1={x2} y1={beamY + 25} x2={x2} y2={dimYChain + 5} stroke="#cbd5e1" strokeWidth="1" strokeDasharray="2 2"/>
                                    <path 
                                        d={`M ${x1},${dimYChain} L ${x2},${dimYChain}`} 
                                        stroke="#94a3b8" 
                                        strokeWidth="1" 
                                        markerStart="url(#dim-arrow-start)" 
                                        markerEnd="url(#dim-arrow-end)" 
                                    />
                                    <rect x={(x1+x2)/2 - 14} y={dimYChain - 8} width="28" height="16" rx="2" fill="white" />
                                    <text x={(x1+x2)/2} y={dimYChain + 4} textAnchor="middle" fontSize="10" fill="#64748b" fontFamily="JetBrains Mono">
                                        {parseFloat(dist.toFixed(3))}
                                    </text>
                                </g>
                            )
                        })}
                    </g>

                    {/* Total Length */}
                    <g opacity="0.8">
                        <line x1={scaleX(0)} y1={dimYChain + 15} x2={scaleX(0)} y2={dimYTotal + 8} stroke="#cbd5e1" strokeWidth="1" strokeDasharray="2 2"/>
                        <line x1={scaleX(length)} y1={dimYChain + 15} x2={scaleX(length)} y2={dimYTotal + 8} stroke="#cbd5e1" strokeWidth="1" strokeDasharray="2 2"/>
                        <path 
                            d={`M ${scaleX(0)},${dimYTotal} L ${scaleX(length)},${dimYTotal}`} 
                            stroke="#94a3b8" 
                            strokeWidth="1" 
                            markerStart="url(#dim-arrow-start)" 
                            markerEnd="url(#dim-arrow-end)" 
                        />
                        <rect x={scaleX(length/2) - 30} y={dimYTotal - 9} width="60" height="18" rx="4" fill="white" stroke="#cbd5e1" strokeWidth="1" />
                        <text x={scaleX(length/2)} y={dimYTotal + 4} textAnchor="middle" fontSize="11" fill="#1e293b" fontWeight="bold" fontFamily="monospace">
                            L = {length}m
                        </text>
                    </g>

                    {/* ILD Marker */}
                    {selectedILDPoint !== undefined && selectedILDPoint !== null && (
                        <g transform={`translate(${scaleX(selectedILDPoint)}, ${beamY})`}>
                            <line x1="0" y1="-70" x2="0" y2="70" stroke="#8b5cf6" strokeWidth="2" strokeDasharray="4" />
                            <circle cx="0" cy="0" r="4" fill="#8b5cf6" stroke="white" strokeWidth="2" />
                            <g transform="translate(0, -85)">
                                <rect x="-40" y="0" width="80" height="20" rx="4" fill="#8b5cf6" />
                                <text x="0" y="14" textAnchor="middle" fontSize="11" fill="white" fontWeight="bold">ILD @ {selectedILDPoint}m</text>
                                <path d="M0,20 L-4,24 L4,24 Z" fill="#8b5cf6" />
                            </g>
                        </g>
                    )}
                </g>
            </svg>
        </div>
    );
};

export default BeamVisualizer;
