import React, { useState, useMemo } from 'react';
import {
    Calculator, Ruler, Settings, Clipboard, Layers, Box, Info,
    ChevronDown, ChevronUp, Hammer
} from 'lucide-react';

// --- Types ---

type Unit = 'metric' | 'imperial';
type InputMode = 'dimensions' | 'area';

// --- Helper Components ---

const Card: React.FC<{ title: string; icon: React.ElementType; children: React.ReactNode; className?: string }> = ({ title, icon: Icon, children, className = "" }) => (
    <div className={`bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-sm shadow-sm overflow-hidden ${className}`}>
        <div className="bg-slate-50 dark:bg-slate-950 px-4 py-3 border-b border-slate-200 dark:border-slate-800 flex items-center gap-2">
            <Icon className="w-4 h-4 text-eng-blue" />
            <h3 className="font-bold text-sm uppercase tracking-wider text-eng-navy dark:text-slate-100">{title}</h3>
        </div>
        <div className="p-4">{children}</div>
    </div>
);

const InputGroup: React.FC<{ label: string; children: React.ReactNode }> = ({ label, children }) => (
    <div className="space-y-1.5 mb-4 last:mb-0">
        <label className="block text-[10px] font-bold text-slate-500 dark:text-slate-400 uppercase tracking-widest">{label}</label>
        {children}
    </div>
);

// --- Main Component ---

export const StoneMasonryCalculator: React.FC = () => {
    // State
    const [unit, setUnit] = useState<Unit>('metric');
    const [mode, setMode] = useState<InputMode>('dimensions');
    const [showBreakdown, setShowBreakdown] = useState(true);

    // Basic Inputs
    const [length, setLength] = useState<string>('10'); // m
    const [height, setHeight] = useState<string>('2'); // m
    const [thickness, setThickness] = useState<string>('0.5'); // m (50cm foundation usually)
    const [totalVolume, setTotalVolume] = useState<string>('10'); // m3

    // Tech Specs (Defaults based on book)
    const [mixRatio, setMixRatio] = useState<string>("1:6"); // 1:4, 1:6 typical for foundation
    const [stoneWastage, setStoneWastage] = useState<string>('15'); // 15-20%
    const [dryMortarPercent, setDryMortarPercent] = useState<string>('40'); // 30-40% of total volume
    const [wcRatio, setWcRatio] = useState<string>('0.6');

    // --- Calculations ---
    const results = useMemo(() => {
        const parseSafe = (val: string) => parseFloat(val) || 0;

        // 1. Calculate Total Work Volume (Quantity)
        let workVol_m3 = 0;
        if (mode === 'dimensions') {
            const l_m = unit === 'imperial' ? parseSafe(length) * 0.3048 : parseSafe(length);
            const h_m = unit === 'imperial' ? parseSafe(height) * 0.3048 : parseSafe(height);
            const t_m = unit === 'imperial' ? parseSafe(thickness) * 0.3048 : parseSafe(thickness);
            workVol_m3 = l_m * h_m * t_m;
        } else {
            workVol_m3 = unit === 'imperial' ? parseSafe(totalVolume) * 0.0283168 : parseSafe(totalVolume);
        }

        // 2. Stone Volume
        // Book: Volume of stone = Work Vol x 1.15 (15% increase)
        const stoneIncreaseFactor = 1 + (parseSafe(stoneWastage) / 100);
        const stoneVol_m3 = workVol_m3 * stoneIncreaseFactor;

        // 3. Dry Mortar Volume
        // Book: Dry mortar = 40% of work quantity
        const dryMortarMsg = parseSafe(dryMortarPercent) / 100;
        const dryMortarVol_m3 = workVol_m3 * dryMortarMsg;

        // 4. Mix
        const [c, s] = mixRatio.split(':').map(Number);
        const totalParts = c + s;

        // Cement
        const cementVol_m3 = (dryMortarVol_m3 * c) / totalParts;
        const cementBags = cementVol_m3 * 28.8; // Book uses 28.8 multiplier (1440/50)

        // Sand
        const sandVol_m3 = (dryMortarVol_m3 * s) / totalParts;
        const sand_cuft = sandVol_m3 * 35.3147;

        // Water
        // Book: 0.6 x Bags x 50
        const waterLiters = (cementBags * 50) * parseSafe(wcRatio);

        // 5. Detailed Breakdown Steps
        const steps: { label: string; value: string; details: string[] }[] = [];

        steps.push({
            label: "1. Total Work Volume",
            value: `${workVol_m3.toFixed(2)} m³`,
            details: mode === 'dimensions'
                ? [
                    `Length = ${unit === 'imperial' ? parseSafe(length) + ' ft' : parseSafe(length) + ' m'}`,
                    `Height = ${unit === 'imperial' ? parseSafe(height) + ' ft' : parseSafe(height) + ' m'}`,
                    `Thickness = ${unit === 'imperial' ? parseSafe(thickness) + ' ft' : parseSafe(thickness) + ' m'}`,
                    `Volume = L × H × T = ${workVol_m3.toFixed(2)} m³`
                ]
                : [`Direct Volume Input = ${totalVolume} ${unit === 'imperial' ? 'cu.ft' : 'm³'}`, `= ${workVol_m3.toFixed(2)} m³ (converted)`]
        });

        steps.push({
            label: "2. Volume of Stone",
            value: `${stoneVol_m3.toFixed(2)} m³`,
            details: [
                `Net Volume = ${workVol_m3.toFixed(2)} m³`,
                `Increase by ${stoneWastage}% for wastage and dressing:`,
                `= ${workVol_m3.toFixed(2)} × (1 + ${parseSafe(stoneWastage) / 100})`,
                `= ${workVol_m3.toFixed(2)} × ${stoneIncreaseFactor}`,
                `= ${stoneVol_m3.toFixed(2)} m³`
            ]
        });

        steps.push({
            label: "3. Volume of Dry Mortar",
            value: `${dryMortarVol_m3.toFixed(2)} m³`,
            details: [
                `Take ${dryMortarPercent}% of total work quantity for irregular shapes:`,
                `= ${dryMortarPercent}% of ${workVol_m3.toFixed(2)} m³`,
                `= ${workVol_m3.toFixed(2)} × ${dryMortarMsg}`,
                `= ${dryMortarVol_m3.toFixed(2)} m³`
            ]
        });

        steps.push({
            label: "4. Cement Quantity",
            value: `${cementBags.toFixed(2)} Bags`,
            details: [
                `Mix Ratio = ${mixRatio} (Total Parts = ${totalParts})`,
                `Cement Volume = Dry Mortar × (1 / ${totalParts})`,
                `= ${dryMortarVol_m3.toFixed(2)} × (1 / ${totalParts}) = ${cementVol_m3.toFixed(4)} m³`,
                `In Bags (× 28.8 bags/m³):`,
                `= ${cementVol_m3.toFixed(4)} × 28.8`,
                `= ${cementBags.toFixed(2)} Bags`
            ]
        });

        steps.push({
            label: "5. Sand Quantity",
            value: `${sandVol_m3.toFixed(2)} m³`,
            details: [
                `Sand Volume = Dry Mortar × (${s} / ${totalParts})`,
                `= ${dryMortarVol_m3.toFixed(2)} × (${s} / ${totalParts})`,
                `= ${sandVol_m3.toFixed(2)} m³`,
                `(= ${sand_cuft.toFixed(2)} cu.ft)`
            ]
        });

        steps.push({
            label: "6. Water Quantity",
            value: `${waterLiters.toFixed(1)} Liters`,
            details: [
                `W/C Ratio = ${wcRatio}`,
                `Cement Weight = ${cementBags.toFixed(2)} Bags × 50 kg = ${(cementBags * 50).toFixed(1)} kg`,
                `Water = Cement Weight × ${wcRatio}`,
                `= ${(cementBags * 50).toFixed(1)} × ${wcRatio}`,
                `= ${waterLiters.toFixed(1)} Liters`
            ]
        });

        return {
            workVol: unit === 'imperial' ? workVol_m3 * 35.3147 : workVol_m3,
            stoneVol: stoneVol_m3,
            cement: cementBags,
            sand_m3: sandVol_m3,
            sand_cf: sand_cuft,
            water: waterLiters,
            steps
        };
    }, [unit, mode, length, height, thickness, totalVolume, mixRatio, stoneWastage, dryMortarPercent, wcRatio]);

    return (
        <div className="grid lg:grid-cols-2 gap-8 items-start">
            {/* Left Panel: Inputs */}
            <div className="space-y-6">
                <div className="flex justify-between items-center">
                    <div className="flex bg-slate-100 dark:bg-slate-800 p-1 rounded-sm">
                        <button onClick={() => setUnit('imperial')} className={`px-4 py-1.5 text-xs font-bold uppercase rounded-sm transition-all ${unit === 'imperial' ? 'bg-white dark:bg-slate-600 shadow-sm text-eng-navy dark:text-white' : 'text-slate-500'}`}>Imperial (ft)</button>
                        <button onClick={() => setUnit('metric')} className={`px-4 py-1.5 text-xs font-bold uppercase rounded-sm transition-all ${unit === 'metric' ? 'bg-white dark:bg-slate-600 shadow-sm text-eng-navy dark:text-white' : 'text-slate-500'}`}>Metric (m)</button>
                    </div>
                </div>

                <Card title="Wall Dimensions" icon={Ruler}>
                    <div className="flex border-b border-slate-200 dark:border-slate-700 mb-6">
                        <button onClick={() => setMode('dimensions')} className={`flex-1 pb-2 text-xs font-bold uppercase tracking-wide transition-colors ${mode === 'dimensions' ? 'text-eng-blue border-b-2 border-eng-blue' : 'text-slate-400 hover:text-slate-600'}`}>Dimensions</button>
                        <button onClick={() => setMode('area')} className={`flex-1 pb-2 text-xs font-bold uppercase tracking-wide transition-colors ${mode === 'area' ? 'text-eng-blue border-b-2 border-eng-blue' : 'text-slate-400 hover:text-slate-600'}`}>Volume Input</button>
                    </div>

                    {mode === 'dimensions' ? (
                        <div className="space-y-4">
                            <InputGroup label={`Length (${unit === 'metric' ? 'm' : 'ft'})`}>
                                <input type="text" value={length} onChange={e => setLength(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" />
                            </InputGroup>
                            <div className="grid grid-cols-2 gap-4">
                                <InputGroup label={`Height (${unit === 'metric' ? 'm' : 'ft'})`}>
                                    <input type="text" value={height} onChange={e => setHeight(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" />
                                </InputGroup>
                                <InputGroup label={`Thickness (${unit === 'metric' ? 'm' : 'ft'})`}>
                                    <input type="text" value={thickness} onChange={e => setThickness(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" />
                                </InputGroup>
                            </div>
                        </div>
                    ) : (
                        <InputGroup label={`Total Volume (${unit === 'metric' ? 'm³' : 'cu.ft'})`}>
                            <input type="text" value={totalVolume} onChange={e => setTotalVolume(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" />
                        </InputGroup>
                    )}
                </Card>

                <Card title="Tech Specs (Standard Norms)" icon={Settings}>
                    <div className="grid grid-cols-2 gap-4">
                        <InputGroup label="Mix Ratio (C:S)">
                            <select value={mixRatio} onChange={e => setMixRatio(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm text-sm">
                                <option value="1:3">1:3 (High Strength)</option>
                                <option value="1:4">1:4 (Standard)</option>
                                <option value="1:5">1:5 (Moderate)</option>
                                <option value="1:6">1:6 (Foundation)</option>
                            </select>
                        </InputGroup>
                        <InputGroup label="W/C Ratio">
                            <input type="text" value={wcRatio} onChange={e => setWcRatio(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" />
                        </InputGroup>
                        <InputGroup label="Dry Mortar %">
                            <div className="relative">
                                <input type="text" value={dryMortarPercent} onChange={e => setDryMortarPercent(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" />
                                <div className="absolute right-2 top-2.5 flex items-center group">
                                    <Info className="w-3 h-3 text-slate-300" />
                                    <div className="absolute bottom-full right-0 mb-2 w-48 p-2 bg-slate-800 text-white text-[9px] rounded-sm opacity-0 group-hover:opacity-100 transition-opacity pointer-events-none z-50">
                                        Standard 30% to 40% of total masonry volume.
                                    </div>
                                </div>
                            </div>
                        </InputGroup>
                        <InputGroup label="Stone Wastage %">
                            <input type="text" value={stoneWastage} onChange={e => setStoneWastage(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" />
                        </InputGroup>
                    </div>
                </Card>
            </div>

            {/* Right Panel: Results */}
            <div className="space-y-6">
                {/* Estimate Result Card */}
                <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-sm shadow-sm">
                    <div className="bg-eng-navy px-4 py-3 flex justify-between items-center">
                        <h3 className="font-bold text-sm uppercase tracking-wide text-white flex items-center gap-2">
                            <Clipboard className="w-4 h-4" /> Bill of Quantities
                        </h3>
                        <span className="bg-white/10 px-2 py-0.5 rounded text-[10px] text-white font-mono uppercase">Stone Masonry</span>
                    </div>

                    <div className="p-0">
                        {/* Summary Metrics */}
                        <div className="bg-eng-blue/5 dark:bg-eng-blue/10 p-4 text-center border-b border-eng-blue/10">
                            <div className="text-[10px] text-eng-blue uppercase font-bold tracking-wider mb-1">Total Work Volume</div>
                            <div className="text-3xl font-mono font-bold text-eng-blue">
                                {results.workVol.toFixed(2)} <span className="text-sm font-sans uppercase font-normal">{unit === 'imperial' ? 'cu.ft' : 'm³'}</span>
                            </div>
                        </div>

                        {/* Material Breakdown */}
                        <div className="p-6 space-y-6">
                            {/* Stone */}
                            <div className="flex justify-between items-center">
                                <div className="flex items-center gap-4">
                                    <div className="w-10 h-10 rounded-full bg-slate-200 dark:bg-slate-700 flex items-center justify-center font-bold text-slate-600 dark:text-slate-300 border border-slate-300 dark:border-slate-600">
                                        <Hammer className="w-5 h-5" />
                                    </div>
                                    <div>
                                        <div className="text-sm font-bold text-eng-navy dark:text-white uppercase tracking-wide">Stone Required</div>
                                        <div className="text-[10px] text-slate-400 font-medium tracking-wider uppercase">
                                            Includes {stoneWastage}% Wastage
                                        </div>
                                    </div>
                                </div>
                                <div className="text-right">
                                    <div className="text-2xl font-mono font-bold text-eng-navy dark:text-white">{results.stoneVol.toFixed(2)} <span className="text-xs font-sans text-slate-400 uppercase">m³</span></div>
                                    <div className="text-[10px] text-slate-400 font-mono italic">
                                        approx {(results.stoneVol / 3).toFixed(1)} truckloads (3m³)
                                    </div>
                                </div>
                            </div>

                            {/* Cement */}
                            <div className="flex justify-between items-center">
                                <div className="flex items-center gap-4">
                                    <div className="w-10 h-10 rounded-full bg-slate-100 dark:bg-slate-800 flex items-center justify-center font-bold text-slate-600 border border-slate-200 dark:border-slate-700">C</div>
                                    <div>
                                        <div className="text-sm font-bold text-eng-navy dark:text-white uppercase tracking-wide">Cement Bags</div>
                                        <div className="text-[10px] text-slate-400 font-medium tracking-wider uppercase">
                                            (50KG Bags)
                                        </div>
                                    </div>
                                </div>
                                <div className="text-right">
                                    <div className="text-2xl font-mono font-bold text-eng-navy dark:text-white">{Math.ceil(results.cement)} <span className="text-xs font-sans text-slate-400 uppercase">bags</span></div>
                                    <div className="text-[10px] text-slate-400 font-mono italic opacity-70">exact: {results.cement.toFixed(2)}</div>
                                </div>
                            </div>

                            {/* Sand */}
                            <div className="flex justify-between items-center">
                                <div className="flex items-center gap-4">
                                    <div className="w-10 h-10 rounded-full bg-yellow-50 dark:bg-yellow-900/10 flex items-center justify-center font-bold text-yellow-600 border border-yellow-100 dark:border-yellow-900/30">S</div>
                                    <div>
                                        <div className="text-sm font-bold text-eng-navy dark:text-white uppercase tracking-wide">Sand</div>
                                        <div className="text-[10px] text-slate-400 font-medium tracking-wider uppercase">
                                            River / Pit Sand
                                        </div>
                                    </div>
                                </div>
                                <div className="text-right">
                                    <div className="text-2xl font-mono font-bold text-eng-navy dark:text-white">{results.sand_cf.toFixed(1)} <span className="text-xs font-sans text-slate-400 uppercase">cu.ft</span></div>
                                    <div className="text-[10px] text-slate-400 font-mono">({results.sand_m3.toFixed(2)} m³)</div>
                                </div>
                            </div>

                            {/* Water */}
                            <div className="flex justify-between items-center bg-blue-50 dark:bg-blue-900/10 p-3 rounded-sm border border-blue-100/50 dark:border-blue-900/30">
                                <div className="flex items-center gap-4">
                                    <div className="w-10 h-10 rounded-full bg-blue-100 dark:bg-blue-900/30 flex items-center justify-center font-bold text-blue-600 border border-blue-200 dark:border-blue-900/50">W</div>
                                    <div>
                                        <div className="text-sm font-bold text-eng-navy dark:text-white uppercase tracking-wide">Water</div>
                                        <div className="text-[10px] text-slate-400 font-medium tracking-wider uppercase">
                                            W/C Ratio: {wcRatio}
                                        </div>
                                    </div>
                                </div>
                                <div className="text-right">
                                    <div className="text-2xl font-mono font-bold text-blue-600">{Math.ceil(results.water)} <span className="text-xs font-sans text-slate-400 uppercase">Liters</span></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Calculation Breakdown */}
                <div className="border border-eng-blue/10 bg-slate-50/30 rounded-sm overflow-hidden">
                    <button
                        onClick={() => setShowBreakdown(!showBreakdown)}
                        className="w-full flex items-center justify-between p-4 bg-slate-50/80 dark:bg-slate-800/80 hover:bg-slate-100 dark:hover:bg-slate-700 transition-colors"
                    >
                        <div className="flex items-center gap-2">
                            <Calculator className="w-4 h-4 text-eng-blue" />
                            <span className="font-bold text-sm uppercase tracking-wide text-eng-navy dark:text-slate-200">Manual Calculation Breakdown</span>
                        </div>
                        {showBreakdown ? <ChevronUp className="w-4 h-4 text-slate-500" /> : <ChevronDown className="w-4 h-4 text-slate-500" />}
                    </button>

                    {showBreakdown && (
                        <div className="p-5 border-t border-slate-100 dark:border-slate-800 bg-white dark:bg-slate-900/50">
                            <div className="space-y-4">
                                {results.steps.map((step, idx) => (
                                    <div key={idx} className="border-b border-slate-100 dark:border-slate-800 last:border-0 pb-6 last:pb-0">
                                        <div className="flex justify-between items-start mb-2">
                                            <h4 className="font-bold text-xs text-eng-navy dark:text-white uppercase tracking-wider opacity-70">{step.label}</h4>
                                            <span className="font-mono text-sm font-bold text-eng-blue bg-eng-blue/5 dark:bg-eng-blue/10 px-2 py-0.5 rounded border border-eng-blue/10">
                                                {step.value}
                                            </span>
                                        </div>
                                        <div className="bg-slate-50 dark:bg-slate-950 p-4 rounded-sm border border-slate-200 dark:border-slate-800 text-xs text-slate-600 dark:text-slate-400 font-mono leading-relaxed space-y-1">
                                            {step.details.map((line, i) => (
                                                <div key={i} className={line.startsWith('=') ? 'pl-4 font-bold text-slate-700 dark:text-slate-300' : ''}>
                                                    {line}
                                                </div>
                                            ))}
                                        </div>
                                    </div>
                                ))}
                            </div>
                        </div>
                    )}
                </div>

            </div>
        </div>
    );
};
