import React, { useState } from 'react';
import { Layers, RefreshCw } from 'lucide-react';

export const SlabEstimator: React.FC = () => {
    const [length, setLength] = useState<string>('0');
    const [width, setWidth] = useState<string>('0');
    const [thickness, setThickness] = useState<string>('5'); // inches
    const [grade, setGrade] = useState<string>("M20");
    const [wcRatio, setWcRatio] = useState<string>('0.5');
    const [result, setResult] = useState<{ cement: number; sand: number; aggregate: number; vol: number; water: number } | null>(null);

    const calculate = () => {
        const parseSafe = (val: string) => parseFloat(val) || 0;

        const L = parseSafe(length);
        const W = parseSafe(width);
        const T = parseSafe(thickness);
        const wc = parseSafe(wcRatio);

        // Wet Volume
        const volCuFt = L * W * (T / 12);
        const volM3 = volCuFt / 35.3147;

        // Dry Volume (Increase by 54%)
        const dryVol = volM3 * 1.54;

        let ratioSum = 0;
        let cementPart = 1;
        let sandPart = 0;
        let aggPart = 0;

        // Standard mix ratios
        switch (grade) {
            case "M15": // 1:2:4
                ratioSum = 1 + 2 + 4; sandPart = 2; aggPart = 4; break;
            case "M20": // 1:1.5:3
                ratioSum = 1 + 1.5 + 3; sandPart = 1.5; aggPart = 3; break;
            case "M25": // 1:1:2
                ratioSum = 1 + 1 + 2; sandPart = 1; aggPart = 2; break;
            default:
                ratioSum = 5.5; sandPart = 1.5; aggPart = 3;
        }

        const cementVol = (dryVol * cementPart) / ratioSum;
        const cementBags = Math.ceil(cementVol * 28.8); // 1440 kg/m3 / 50kg bag

        const sandVol = (dryVol * sandPart) / ratioSum;
        const sandCuFt = parseFloat((sandVol * 35.3147).toFixed(2));

        const aggVol = (dryVol * aggPart) / ratioSum;
        const aggCuFt = parseFloat((aggVol * 35.3147).toFixed(2));

        const waterLiters = Math.ceil((cementBags * 50) * wc);

        setResult({
            vol: parseFloat(volCuFt.toFixed(2)),
            cement: cementBags,
            sand: sandCuFt,
            aggregate: aggCuFt,
            water: waterLiters
        });
    };

    return (
        <div className="bg-white dark:bg-slate-900 p-6 rounded-lg shadow-sm border border-slate-200 dark:border-slate-800">
            <h3 className="text-xl font-bold text-eng-navy dark:text-white mb-4 flex items-center gap-2">
                <Layers className="w-5 h-5 text-eng-blue" /> RCC Slab Calculator
            </h3>

            <div className="grid md:grid-cols-2 gap-6">
                <div className="space-y-4">
                    <div>
                        <label className="block text-sm font-medium text-slate-700 dark:text-slate-300 mb-1">Length (ft)</label>
                        <input
                            type="text"
                            value={length}
                            onChange={(e) => setLength(e.target.value)}
                            className="w-full p-2 border border-slate-300 dark:border-slate-700 rounded bg-slate-50 dark:bg-slate-800"
                            placeholder="e.g. 25"
                        />
                    </div>
                    <div>
                        <label className="block text-sm font-medium text-slate-700 dark:text-slate-300 mb-1">Width (ft)</label>
                        <input
                            type="text"
                            value={width}
                            onChange={(e) => setWidth(e.target.value)}
                            className="w-full p-2 border border-slate-300 dark:border-slate-700 rounded bg-slate-50 dark:bg-slate-800"
                            placeholder="e.g. 15"
                        />
                    </div>
                    <div>
                        <label className="block text-sm font-medium text-slate-700 dark:text-slate-300 mb-1">Thickness (inch)</label>
                        <input
                            type="text"
                            value={thickness}
                            onChange={(e) => setThickness(e.target.value)}
                            className="w-full p-2 border border-slate-300 dark:border-slate-700 rounded bg-slate-50 dark:bg-slate-800"
                            placeholder="e.g. 5"
                        />
                    </div>
                    <div>
                        <label className="block text-sm font-medium text-slate-700 dark:text-slate-300 mb-1">Concrete Grade</label>
                        <select
                            value={grade}
                            onChange={(e) => setGrade(e.target.value)}
                            className="w-full p-2 border border-slate-300 dark:border-slate-700 rounded bg-slate-50 dark:bg-slate-800"
                        >
                            <option value="M15">M15 (1:2:4)</option>
                            <option value="M20">M20 (1:1.5:3)</option>
                            <option value="M25">M25 (1:1:2)</option>
                        </select>
                    </div>
                    <div>
                        <label className="block text-sm font-medium text-slate-700 dark:text-slate-300 mb-1">W/C Ratio</label>
                        <input
                            type="text"
                            value={wcRatio}
                            onChange={(e) => setWcRatio(e.target.value)}
                            className="w-full p-2 border border-slate-300 dark:border-slate-700 rounded bg-slate-50 dark:bg-slate-800 font-mono"
                            placeholder="e.g. 0.5"
                        />
                    </div>
                    <button
                        onClick={calculate}
                        className="w-full py-3 bg-eng-navy hover:bg-eng-blue text-white font-bold rounded transition-colors"
                    >
                        Calculate
                    </button>
                </div>

                <div className="bg-slate-50 dark:bg-slate-950 p-6 rounded border border-slate-100 dark:border-slate-800 flex flex-col justify-center">
                    {result ? (
                        <div className="space-y-4">
                            <h4 className="font-bold text-lg border-b border-slate-200 dark:border-slate-700 pb-2">Material Breakdown</h4>
                            <div className="flex justify-between items-center text-sm">
                                <span className="text-slate-500">Total Volume</span>
                                <span className="font-mono">{result.vol} Cu.ft</span>
                            </div>
                            <div className="flex justify-between items-center">
                                <span className="text-slate-600 dark:text-slate-400">Cement (Bags)</span>
                                <span className="text-2xl font-bold text-eng-blue">{result.cement}</span>
                            </div>
                            <div className="flex justify-between items-center">
                                <span className="text-slate-600 dark:text-slate-400">Sand (Cu.ft)</span>
                                <span className="text-2xl font-bold text-eng-blue">{result.sand}</span>
                            </div>
                            <div className="flex justify-between items-center">
                                <span className="text-slate-600 dark:text-slate-400">Aggregate (Cu.ft)</span>
                                <span className="text-2xl font-bold text-eng-blue">{result.aggregate}</span>
                            </div>
                            <div className="flex justify-between items-center bg-blue-50 dark:bg-blue-900/10 p-2 rounded-sm border border-blue-100 dark:border-blue-900/30">
                                <span className="text-blue-700 dark:text-blue-400 font-bold">Water (Liters)</span>
                                <span className="text-2xl font-bold text-blue-600">{result.water}</span>
                            </div>
                        </div>
                    ) : (
                        <div className="text-center text-slate-400">
                            <RefreshCw className="w-12 h-12 mx-auto mb-2 opacity-20" />
                            <p>Enter dimensions to estimate concrete</p>
                        </div>
                    )}
                </div>
            </div>
        </div>
    );
};
