import React, { useState, useMemo } from 'react';
import {
    Calculator, Ruler, Settings, Clipboard, Plus, Trash2, Layers, Box, Info,
    ChevronDown, ChevronUp
} from 'lucide-react';

// --- Types ---

type Unit = 'metric' | 'imperial';
type InputMode = 'dimensions' | 'area';

interface Deduction {
    id: string;
    type: string;
    width: string;
    height: string;
    count: string;
}

// --- Helper Components ---

const Card: React.FC<{ title: string; icon: React.ElementType; children: React.ReactNode; className?: string }> = ({ title, icon: Icon, children, className = "" }) => (
    <div className={`bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-sm shadow-sm overflow-hidden ${className}`}>
        <div className="bg-slate-50 dark:bg-slate-950 px-4 py-3 border-b border-slate-200 dark:border-slate-800 flex items-center gap-2">
            <Icon className="w-4 h-4 text-eng-blue" />
            <h3 className="font-bold text-sm uppercase tracking-wider text-eng-navy dark:text-slate-100">{title}</h3>
        </div>
        <div className="p-4">{children}</div>
    </div>
);

const InputGroup: React.FC<{ label: string; children: React.ReactNode }> = ({ label, children }) => (
    <div className="space-y-1.5 mb-4 last:mb-0">
        <label className="block text-[10px] font-bold text-slate-500 dark:text-slate-400 uppercase tracking-widest">{label}</label>
        {children}
    </div>
);

// --- Main Component ---

export const PlasterEstimator: React.FC = () => {
    // State
    const [unit, setUnit] = useState<Unit>('metric');
    const [mode, setMode] = useState<InputMode>('dimensions');
    const [showBreakdown, setShowBreakdown] = useState(true);

    // Basic Inputs
    const [length, setLength] = useState<string>('10');
    const [height, setHeight] = useState<string>('3');
    const [totalArea, setTotalArea] = useState<string>('30');
    const [thickness, setThickness] = useState<string>('12'); // mm for metric, will convert
    const [customThickness, setCustomThickness] = useState<string>('12');

    // Tech Specs
    const [mixRatio, setMixRatio] = useState<string>("1:4"); // 1:3, 1:4, 1:5, 1:6
    const [wastage, setWastage] = useState<string>('5');
    const [dryVolFactor, setDryVolFactor] = useState<string>('1.33'); // Standard for plaster
    const [wcRatio, setWcRatio] = useState<string>('0.5');
    const [irregularity, setIrregularity] = useState<string>('25'); // 25% for joints/irregular surface

    // Deductions
    const [deductions, setDeductions] = useState<Deduction[]>([]);

    // --- Actions ---
    const addDeduction = () => {
        setDeductions([...deductions, {
            id: Math.random().toString(36).substr(2, 9),
            type: 'Door/Window',
            width: unit === 'metric' ? '1.2' : '4',
            height: unit === 'metric' ? '2.1' : '7',
            count: '1'
        }]);
    };

    const removeDeduction = (id: string) => {
        setDeductions(deductions.filter(d => d.id !== id));
    };

    const updateDeduction = (id: string, field: keyof Deduction, value: any) => {
        setDeductions(prev => prev.map(d => d.id === id ? { ...d, [field]: value } : d));
    };

    // --- Calculations ---
    const results = useMemo(() => {
        const parseSafe = (val: string) => parseFloat(val) || 0;

        // 1. Calculate Gross Area
        let grossArea_m2 = 0;
        if (mode === 'dimensions') {
            const l_m = unit === 'imperial' ? parseSafe(length) * 0.3048 : parseSafe(length);
            const h_m = unit === 'imperial' ? parseSafe(height) * 0.3048 : parseSafe(height);
            grossArea_m2 = l_m * h_m;
        } else {
            grossArea_m2 = unit === 'imperial' ? parseSafe(totalArea) * 0.092903 : parseSafe(totalArea);
        }

        // 2. Total Deductions
        let deductionArea_m2 = 0;
        deductions.forEach(d => {
            const w_m = unit === 'imperial' ? parseSafe(d.width) * 0.3048 : parseSafe(d.width);
            const h_m = unit === 'imperial' ? parseSafe(d.height) * 0.3048 : parseSafe(d.height);
            deductionArea_m2 += (w_m * h_m * parseSafe(d.count));
        });

        const netArea_m2 = Math.max(0, grossArea_m2 - deductionArea_m2);

        // 3. Volume
        const t_m = unit === 'imperial' ? (parseSafe(thickness) * 0.0254) : (parseSafe(thickness) / 1000);

        const wetVol_m3 = (netArea_m2 * t_m) * (1 + parseSafe(irregularity) / 100);
        const dryVol_m3 = wetVol_m3 * parseSafe(dryVolFactor) * (1 + parseSafe(wastage) / 100);

        // 4. Mix
        const [c, s] = mixRatio.split(':').map(Number);
        const totalParts = c + s;

        const cementVol_m3 = (dryVol_m3 * c) / totalParts;
        const cementBags = (cementVol_m3 * 1440) / 50; // 1440 kg/m3 density, 50kg per bag

        const sandVol_m3 = (dryVol_m3 * s) / totalParts;
        const sand_cuft = sandVol_m3 * 35.3147;

        // 5. Steps for Breakdown
        // 5. Detailed Breakdown Steps
        const steps: { label: string; value: string; details: string[] }[] = [];

        steps.push({
            label: "1. Surface Area",
            value: `${grossArea_m2.toFixed(2)} m²`,
            details: mode === 'dimensions'
                ? [
                    `Length = ${unit === 'imperial' ? parseSafe(length) + ' ft' : parseSafe(length) + ' m'}`,
                    `Height = ${unit === 'imperial' ? parseSafe(height) + ' ft' : parseSafe(height) + ' m'}`,
                    `Area = L × H = ${(unit === 'imperial' ? parseSafe(length) * 0.3048 : parseSafe(length)).toFixed(2)} × ${(unit === 'imperial' ? parseSafe(height) * 0.3048 : parseSafe(height)).toFixed(2)}`,
                    `= ${grossArea_m2.toFixed(2)} m²`
                ]
                : [`Direct Area Input = ${totalArea} ${unit === 'imperial' ? 'sq.ft' : 'm²'}`, `= ${grossArea_m2.toFixed(2)} m² (converted)`]
        });

        if (deductions.length > 0) {
            steps.push({
                label: "2. Deduction Area",
                value: `${deductionArea_m2.toFixed(2)} m²`,
                details: [
                    "Sum of all openings:",
                    ...deductions.map(d => `${d.type}: ${parseSafe(d.width)}x${parseSafe(d.height)} @ ${parseSafe(d.count)} nos = ${(parseSafe(d.width) * parseSafe(d.height) * parseSafe(d.count)).toFixed(2)} m²`),
                    `Total Deduction = ${deductionArea_m2.toFixed(2)} m²`
                ]
            });
        }

        const baseWetVol = netArea_m2 * t_m;
        const irregularityVol = baseWetVol * (parseSafe(irregularity) / 100);

        steps.push({
            label: (deductions.length > 0 ? "3. " : "2. ") + "Total Wet Mix Volume",
            value: `${wetVol_m3.toFixed(4)} m³`,
            details: [
                `Base Volume = Net Area × Thickness`,
                `= ${netArea_m2.toFixed(2)} m² × ${t_m.toFixed(3)} m = ${baseWetVol.toFixed(4)} m³`,
                `Add ${irregularity}% for depression, joints, and wastage:`,
                `= ${baseWetVol.toFixed(4)} + (${baseWetVol.toFixed(4)} × ${parseSafe(irregularity) / 100})`,
                `= ${baseWetVol.toFixed(4)} + ${irregularityVol.toFixed(4)}`,
                `= ${wetVol_m3.toFixed(4)} m³`
            ]
        });

        const dryVolNoWastage = wetVol_m3 * parseSafe(dryVolFactor);

        steps.push({
            label: (deductions.length > 0 ? "4. " : "3. ") + "Dry Mortar Volume",
            value: `${dryVol_m3.toFixed(3)} m³`,
            details: [
                `Convert to Dry Volume (Factor ${dryVolFactor}):`,
                `= Wet Volume × ${dryVolFactor}`,
                `= ${wetVol_m3.toFixed(4)} × ${dryVolFactor} = ${dryVolNoWastage.toFixed(4)} m³`,
                `Add ${wastage}% Safety/Wastage:`,
                `= ${dryVolNoWastage.toFixed(4)} × (1 + ${parseSafe(wastage) / 100})`,
                `= ${dryVol_m3.toFixed(4)} m³`
            ]
        });

        steps.push({
            label: (deductions.length > 0 ? "5. " : "4. ") + "Cement Quantity",
            value: `${cementBags.toFixed(1)} Bags`,
            details: [
                `Ratio = ${mixRatio} (Cement : Sand) → Total Parts = ${totalParts}`,
                `Cement Volume = Dry Volume × (C / Total)`,
                `= ${dryVol_m3.toFixed(4)} × (1 / ${totalParts}) = ${cementVol_m3.toFixed(4)} m³`,
                `Convert to Bags (Density 1440 kg/m³, 50kg/bag):`,
                `= (${cementVol_m3.toFixed(4)} × 1440) / 50`,
                `= ${cementBags.toFixed(2)} Bags`
            ]
        });

        steps.push({
            label: (deductions.length > 0 ? "6. " : "5. ") + "Sand Quantity",
            value: `${sand_cuft.toFixed(1)} cu.ft`,
            details: [
                `Sand Volume = Dry Volume × (S / Total)`,
                `= ${dryVol_m3.toFixed(4)} × (${s} / ${totalParts}) = ${sandVol_m3.toFixed(4)} m³`,
                `Convert to Cubic Feet (× 35.3147):`,
                `= ${sandVol_m3.toFixed(4)} × 35.3147`,
                `= ${sand_cuft.toFixed(2)} cu.ft`
            ]
        });

        const waterLiters = (cementBags * 50) * parseSafe(wcRatio);
        steps.push({
            label: (deductions.length > 0 ? "7. " : "6. ") + "Water Quantity",
            value: `${waterLiters.toFixed(1)} Liters`,
            details: [
                `Based on Cement Weight (W/C Ratio ${wcRatio}):`,
                `Cement Weight = ${cementBags.toFixed(2)} bags × 50 kg = ${(cementBags * 50).toFixed(2)} kg`,
                `Water = Cement Weight × ${wcRatio}`,
                `= ${(cementBags * 50).toFixed(2)} × ${wcRatio}`,
                `= ${waterLiters.toFixed(2)} Liters`
            ]
        });

        return {
            grossArea: unit === 'imperial' ? grossArea_m2 * 10.7639 : grossArea_m2,
            netArea: unit === 'imperial' ? netArea_m2 * 10.7639 : netArea_m2,
            wetVol: wetVol_m3,
            dryVol: dryVol_m3,
            cement: cementBags,
            sand_m3: sandVol_m3,
            sand_cf: sand_cuft,
            water: waterLiters,
            deductionArea: unit === 'imperial' ? deductionArea_m2 * 10.7639 : deductionArea_m2,
            steps
        };
    }, [unit, mode, length, height, totalArea, thickness, mixRatio, wastage, dryVolFactor, wcRatio, deductions, irregularity]);

    return (
        <div className="grid lg:grid-cols-2 gap-8 items-start">
            {/* Left Panel: Inputs */}
            <div className="space-y-6">
                <div className="flex justify-between items-center">
                    <div className="flex bg-slate-100 dark:bg-slate-800 p-1 rounded-sm">
                        <button onClick={() => setUnit('imperial')} className={`px-4 py-1.5 text-xs font-bold uppercase rounded-sm transition-all ${unit === 'imperial' ? 'bg-white dark:bg-slate-600 shadow-sm text-eng-navy dark:text-white' : 'text-slate-500'}`}>Imperial (ft)</button>
                        <button onClick={() => setUnit('metric')} className={`px-4 py-1.5 text-xs font-bold uppercase rounded-sm transition-all ${unit === 'metric' ? 'bg-white dark:bg-slate-600 shadow-sm text-eng-navy dark:text-white' : 'text-slate-500'}`}>Metric (m)</button>
                    </div>
                </div>

                <Card title="Surface Integration" icon={Ruler}>
                    <div className="flex border-b border-slate-200 dark:border-slate-700 mb-6">
                        <button onClick={() => setMode('dimensions')} className={`flex-1 pb-2 text-xs font-bold uppercase tracking-wide transition-colors ${mode === 'dimensions' ? 'text-eng-blue border-b-2 border-eng-blue' : 'text-slate-400 hover:text-slate-600'}`}>Dimensions</button>
                        <button onClick={() => setMode('area')} className={`flex-1 pb-2 text-xs font-bold uppercase tracking-wide transition-colors ${mode === 'area' ? 'text-eng-blue border-b-2 border-eng-blue' : 'text-slate-400 hover:text-slate-600'}`}>Direct Area</button>
                    </div>

                    {mode === 'dimensions' ? (
                        <div className="grid grid-cols-2 gap-4">
                            <InputGroup label={`Length (${unit === 'metric' ? 'm' : 'ft'})`}>
                                <input type="text" value={length} onChange={e => setLength(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" />
                            </InputGroup>
                            <InputGroup label={`Height (${unit === 'metric' ? 'm' : 'ft'})`}>
                                <input type="text" value={height} onChange={e => setHeight(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" />
                            </InputGroup>
                        </div>
                    ) : (
                        <InputGroup label={`Total Area (${unit === 'metric' ? 'm²' : 'ft²'})`}>
                            <input type="text" value={totalArea} onChange={e => setTotalArea(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" />
                        </InputGroup>
                    )}

                    <div className="mt-4">
                        <InputGroup label={`Plaster Thickness (${unit === 'metric' ? 'mm' : 'inch'})`}>
                            <div className="grid grid-cols-4 gap-2">
                                {(unit === 'metric' ? [6, 10, 12, 15, 20] : [0.25, 0.375, 0.5, 0.625, 0.75]).map(t => (
                                    <button
                                        key={t}
                                        onClick={() => setThickness(t.toString())}
                                        className={`py-2 text-[10px] font-bold border rounded-sm transition-all ${parseFloat(thickness) === t ? 'bg-eng-blue text-white border-eng-blue' : 'bg-slate-50 dark:bg-slate-800 border-slate-200 dark:border-slate-700 text-slate-500'}`}
                                    >
                                        {t}
                                    </button>
                                ))}
                                <div className="col-span-4 mt-2">
                                    <input
                                        type="text"
                                        placeholder="Custom thickness..."
                                        value={thickness}
                                        onChange={e => setThickness(e.target.value)}
                                        className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-xs"
                                    />
                                </div>
                            </div>
                        </InputGroup>
                    </div>
                </Card>

                <Card title="Deductions" icon={Layers}>
                    <div className="space-y-3">
                        {deductions.map(d => (
                            <div key={d.id} className="flex gap-2 items-center bg-slate-50 dark:bg-slate-800/50 p-2 border border-slate-200 dark:border-slate-700 rounded-sm">
                                <span className="text-[10px] font-bold text-slate-400 w-16 truncate">{d.type}</span>
                                <input type="text" value={d.width} onChange={e => updateDeduction(d.id, 'width', e.target.value)} className="w-full p-1 border rounded-sm text-xs font-mono dark:bg-slate-800 dark:border-slate-700" placeholder="W" />
                                <span className="text-slate-400 text-xs">x</span>
                                <input type="text" value={d.height} onChange={e => updateDeduction(d.id, 'height', e.target.value)} className="w-full p-1 border rounded-sm text-xs font-mono dark:bg-slate-800 dark:border-slate-700" placeholder="H" />
                                <span className="text-slate-400 text-xs">@</span>
                                <input type="text" value={d.count} onChange={e => updateDeduction(d.id, 'count', e.target.value)} className="w-12 p-1 border rounded-sm text-xs font-mono dark:bg-slate-800 dark:border-slate-700" />
                                <button onClick={() => removeDeduction(d.id)} className="text-red-500 hover:bg-red-50 dark:hover:bg-red-900/10 p-1 rounded-sm">
                                    <Trash2 className="w-3.5 h-3.5" />
                                </button>
                            </div>
                        ))}
                        <button onClick={addDeduction} className="w-full py-2 border border-dashed border-slate-300 dark:border-slate-700 text-slate-500 text-xs font-bold uppercase hover:bg-slate-50 dark:hover:bg-slate-800 flex items-center justify-center gap-2 rounded-sm">
                            <Plus className="w-3.5 h-3.5" /> Add Deduction
                        </button>
                    </div>
                </Card>

                <Card title="Technical Specs" icon={Settings}>
                    <div className="grid grid-cols-2 gap-4">
                        <InputGroup label="Mix Ratio (C:S)">
                            <select value={mixRatio} onChange={e => setMixRatio(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm text-sm">
                                <option value="1:3">1:3 (Ceiling)</option>
                                <option value="1:4">1:4 (Internal)</option>
                                <option value="1:5">1:5 (Common)</option>
                                <option value="1:6">1:6 (External/Rough)</option>
                            </select>
                        </InputGroup>
                        <InputGroup label="Dry Vol Factor">
                            <div className="relative">
                                <input type="text" value={dryVolFactor} onChange={e => setDryVolFactor(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" />
                                <div className="absolute right-2 top-2.5 flex items-center group">
                                    <Info className="w-3 h-3 text-slate-300" />
                                    <div className="absolute bottom-full right-0 mb-2 w-48 p-2 bg-slate-800 text-white text-[9px] rounded-sm opacity-0 group-hover:opacity-100 transition-opacity pointer-events-none z-50">
                                        Standard 1.33 factor accounts for voids in sand and mortar filling joints.
                                    </div>
                                </div>
                            </div>
                        </InputGroup>
                        <InputGroup label="Wastage (%)">
                            <input type="text" value={wastage} onChange={e => setWastage(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" />
                        </InputGroup>
                        <InputGroup label="Irregularity (%)">
                            <input type="text" value={irregularity} onChange={e => setIrregularity(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" />
                        </InputGroup>
                        <InputGroup label="W/C Ratio">
                            <input type="text" value={wcRatio} onChange={e => setWcRatio(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" />
                        </InputGroup>
                    </div>
                </Card>
            </div>

            {/* Right Panel: Results */}
            <div className="space-y-6">

                {/* Visual Blueprint */}
                <div className="bg-eng-navy p-1 rounded-sm shadow-md">
                    <div className="bg-[#f0f4f8] relative h-[300px] w-full rounded-sm overflow-hidden flex items-center justify-center">
                        <div className="absolute top-3 left-3 px-2 py-1 bg-white/80 backdrop-blur-sm border border-slate-200 text-[10px] font-bold uppercase tracking-widest text-slate-500 rounded-sm z-10">
                            {mode === 'dimensions' ? 'Surface Blueprint' : 'Direct Area Mode'}
                        </div>

                        {mode === 'dimensions' ? (
                            <div className="relative p-4 w-full h-full flex items-center justify-center">
                                {(() => {
                                    const parseSafe = (val: string) => parseFloat(val) || 0;
                                    const L = unit === 'imperial' ? parseSafe(length) : parseSafe(length);
                                    const H = unit === 'imperial' ? parseSafe(height) : parseSafe(height);

                                    if (L <= 0 || H <= 0) return (
                                        <div className="text-slate-400 text-xs italic font-mono">Enter dimensions to preview...</div>
                                    );

                                    const padX = L * 0.15;
                                    const padY = H * 0.15;
                                    const viewBoxStr = `${-padX} ${-padY} ${L + padX * 2} ${H + padY * 2}`;

                                    return (
                                        <svg width="100%" height="100%" viewBox={viewBoxStr} preserveAspectRatio="xMidYMid meet" className="max-w-full max-h-full drop-shadow-lg">
                                            <defs>
                                                <pattern id="plasterGrid" patternUnits="userSpaceOnUse" width={L / 10} height={H / 10}>
                                                    <path d={`M ${L / 10} 0 L 0 0 0 ${H / 10}`} fill="none" stroke="#e2e8f0" strokeWidth={L * 0.002} />
                                                </pattern>
                                                <marker id="arrowHead" markerWidth="10" markerHeight="10" refX="5" refY="5" orient="auto">
                                                    <path d="M0,0 L10,5 L0,10 Z" fill="#94a3b8" />
                                                </marker>
                                            </defs>

                                            {/* Main Surface */}
                                            <rect x="0" y="0" width={L} height={H} fill="#f8fafc" stroke="#cbd5e1" strokeWidth={L * 0.005} />
                                            <rect x="0" y="0" width={L} height={H} fill="url(#plasterGrid)" />

                                            {/* Deductions */}
                                            {deductions.map((d, i) => {
                                                const dW = unit === 'imperial' ? parseSafe(d.width) : parseSafe(d.width);
                                                const dH = unit === 'imperial' ? parseSafe(d.height) : parseSafe(d.height);
                                                const dCount = parseSafe(d.count);

                                                return Array.from({ length: Math.min(dCount, 10) }).map((_, idx) => {
                                                    const spacing = L / (Math.min(dCount, 10) + 1);
                                                    const xPos = spacing * (idx + 1) - (dW / 2);
                                                    const yPos = H * 0.2; // 20% from top

                                                    return (
                                                        <g key={`${d.id}-${idx}`}>
                                                            <rect
                                                                x={xPos} y={yPos} width={dW} height={dH}
                                                                fill="white" stroke="#3b82f6" strokeWidth={L * 0.005}
                                                                strokeDasharray={`${L * 0.02} ${L * 0.01}`}
                                                            />
                                                            <text x={xPos + dW / 2} y={yPos + dH / 2} textAnchor="middle" dominantBaseline="middle" fontSize={L * 0.04} fill="#3b82f6" fontWeight="bold">
                                                                D{i + 1}
                                                            </text>
                                                        </g>
                                                    );
                                                });
                                            })}

                                            {/* Dimensions */}
                                            <line x1="0" y1={H + padY / 2} x2={L} y2={H + padY / 2} stroke="#94a3b8" strokeWidth={L * 0.005} markerStart="url(#arrowHead)" markerEnd="url(#arrowHead)" />
                                            <text x={L / 2} y={H + padY} textAnchor="middle" fontSize={L * 0.06} fill="#64748b" fontWeight="bold">
                                                {length} {unit === 'imperial' ? 'ft' : 'm'}
                                            </text>

                                            <line x1={L + padX / 2} y1="0" x2={L + padX / 2} y2={H} stroke="#94a3b8" strokeWidth={L * 0.005} markerStart="url(#arrowHead)" markerEnd="url(#arrowHead)" />
                                            <text x={L + padX} y={H / 2} transform={`rotate(90, ${L + padX}, ${H / 2})`} textAnchor="middle" fontSize={L * 0.06} fill="#64748b" fontWeight="bold">
                                                {height} {unit === 'imperial' ? 'ft' : 'm'}
                                            </text>
                                        </svg>
                                    );
                                })()}
                            </div>
                        ) : (
                            <div className="flex flex-col items-center justify-center p-8 text-slate-400 gap-2">
                                <Box className="w-12 h-12 text-slate-300" />
                                <p className="text-xs font-mono uppercase tracking-widest text-center">Batch Processing<br />Direct Area Input</p>
                            </div>
                        )}
                    </div>
                </div>

                {/* Estimate Result Card (BOQ style) */}
                <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-sm shadow-sm">
                    <div className="bg-eng-navy px-4 py-3 flex justify-between items-center">
                        <h3 className="font-bold text-sm uppercase tracking-wide text-white flex items-center gap-2">
                            <Clipboard className="w-4 h-4" /> Final Bill of Quantities
                        </h3>
                        <span className="bg-white/10 px-2 py-0.5 rounded text-[10px] text-white font-mono uppercase">Plaster Est.</span>
                    </div>

                    <div className="p-0">
                        {/* Summary Metrics */}
                        <div className="grid grid-cols-2 gap-px bg-slate-200 dark:bg-slate-800 border-b border-slate-200 dark:border-slate-800">
                            <div className="bg-slate-50/50 dark:bg-slate-900/50 p-4">
                                <div className="text-[10px] text-slate-400 uppercase font-bold tracking-wider mb-1">Gross Surface</div>
                                <div className="text-xl font-mono font-bold text-slate-700 dark:text-slate-200">
                                    {results.grossArea.toFixed(2)} <span className="text-xs font-sans text-slate-400 uppercase font-normal">{unit === 'imperial' ? 'sq.ft' : 'm²'}</span>
                                </div>
                            </div>
                            <div className="bg-slate-50/50 dark:bg-slate-900/50 p-4 border-l border-slate-200 dark:border-slate-800">
                                <div className="text-[10px] text-red-400 uppercase font-bold tracking-wider mb-1">Deduction</div>
                                <div className="text-xl font-mono font-bold text-red-500 dark:text-red-400">
                                    -{results.deductionArea.toFixed(2)} <span className="text-xs font-sans text-red-300 dark:text-red-900/50 uppercase font-normal">{unit === 'imperial' ? 'sq.ft' : 'm²'}</span>
                                </div>
                            </div>
                        </div>

                        {/* Net Result Highlight */}
                        <div className="bg-eng-blue/5 dark:bg-eng-blue/10 p-4 text-center border-b border-eng-blue/10">
                            <div className="text-[10px] text-eng-blue uppercase font-bold tracking-wider mb-1">Net Working Surface</div>
                            <div className="text-3xl font-mono font-bold text-eng-blue">
                                {results.netArea.toFixed(2)} <span className="text-sm font-sans uppercase font-normal">{unit === 'imperial' ? 'sq.ft' : 'm²'}</span>
                            </div>
                        </div>

                        {/* Material Breakdown */}
                        <div className="p-6 space-y-6">
                            {/* Cement */}
                            <div className="flex justify-between items-center">
                                <div className="flex items-center gap-4">
                                    <div className="w-10 h-10 rounded-full bg-slate-100 dark:bg-slate-800 flex items-center justify-center font-bold text-slate-600 border border-slate-200 dark:border-slate-700">C</div>
                                    <div>
                                        <div className="text-sm font-bold text-eng-navy dark:text-white uppercase tracking-wide">Cement Bags</div>
                                        <div className="text-[10px] text-slate-400 font-medium tracking-wider uppercase flex items-center gap-1.5">
                                            <span className="w-1 h-1 bg-slate-300 rounded-full"></span> OPC / PPC (50KG)
                                        </div>
                                    </div>
                                </div>
                                <div className="text-right">
                                    <div className="text-2xl font-mono font-bold text-eng-navy dark:text-white">{Math.ceil(results.cement)} <span className="text-xs font-sans text-slate-400 uppercase">bags</span></div>
                                    <div className="text-[10px] text-slate-400 font-mono italic opacity-70">exact: {results.cement.toFixed(2)}</div>
                                </div>
                            </div>

                            {/* Sand */}
                            <div className="flex justify-between items-center">
                                <div className="flex items-center gap-4">
                                    <div className="w-10 h-10 rounded-full bg-yellow-50 dark:bg-yellow-900/10 flex items-center justify-center font-bold text-yellow-600 border border-yellow-100 dark:border-yellow-900/30">S</div>
                                    <div>
                                        <div className="text-sm font-bold text-eng-navy dark:text-white uppercase tracking-wide">Sand Capacity</div>
                                        <div className="text-[10px] text-slate-400 font-medium tracking-wider uppercase flex items-center gap-1.5">
                                            <span className="w-1 h-1 bg-yellow-300 rounded-full"></span> River / M-Sand
                                        </div>
                                    </div>
                                </div>
                                <div className="text-right">
                                    <div className="text-2xl font-mono font-bold text-eng-navy dark:text-white">{results.sand_cf.toFixed(1)} <span className="text-xs font-sans text-slate-400 uppercase">cu.ft</span></div>
                                    <div className="text-[10px] text-slate-400 font-mono">({results.sand_m3.toFixed(3)} m³)</div>
                                </div>
                            </div>

                            {/* Water */}
                            <div className="flex justify-between items-center bg-blue-50 dark:bg-blue-900/10 p-3 rounded-sm border border-blue-100/50 dark:border-blue-900/30">
                                <div className="flex items-center gap-4">
                                    <div className="w-10 h-10 rounded-full bg-blue-100 dark:bg-blue-900/30 flex items-center justify-center font-bold text-blue-600 border border-blue-200 dark:border-blue-900/50">W</div>
                                    <div>
                                        <div className="text-sm font-bold text-eng-navy dark:text-white uppercase tracking-wide">Water Mix</div>
                                        <div className="text-[10px] text-slate-400 font-medium tracking-wider uppercase flex items-center gap-1.5">
                                            <span className="w-1 h-1 bg-blue-400 rounded-full"></span> Potable Water
                                        </div>
                                    </div>
                                </div>
                                <div className="text-right">
                                    <div className="text-2xl font-mono font-bold text-blue-600">{Math.ceil(results.water)} <span className="text-xs font-sans text-slate-400 uppercase">Liters</span></div>
                                    <div className="text-[10px] text-slate-400 font-mono italic opacity-70">W/C Ratio: {wcRatio}</div>
                                </div>
                            </div>
                        </div>

                        {/* Action Bar */}
                        <div className="bg-slate-50 dark:bg-slate-950 p-4 border-t border-slate-200 dark:border-slate-800 text-center flex gap-2">
                            <button
                                onClick={() => {
                                    const text = `Plaster Estimation Results:\n--------------------------\nNet Area: ${results.netArea.toFixed(2)} ${unit === 'metric' ? 'm2' : 'ft2'}\nCement: ${Math.ceil(results.cement)} Bags\nSand: ${results.sand_cf.toFixed(2)} Cu.ft\nWater: ${Math.ceil(results.water)} Liters\nThickness: ${thickness} ${unit === 'imperial' ? 'inch' : 'mm'}\nMix Ratio: ${mixRatio}\nW/C Ratio: ${wcRatio}`;
                                    navigator.clipboard.writeText(text);
                                    alert("Estimate copied to clipboard!");
                                }}
                                className="flex-1 py-2.5 bg-white dark:bg-slate-800 border border-slate-300 dark:border-slate-700 text-slate-600 dark:text-slate-300 text-xs font-bold uppercase tracking-wider hover:bg-slate-50 dark:hover:bg-slate-700 transition-colors flex items-center justify-center gap-2 rounded-sm shadow-sm"
                            >
                                <Clipboard className="w-3.5 h-3.5" /> Copy Detail
                            </button>
                        </div>
                    </div>
                </div>

                {/* Calculation Breakdown */}
                <div className="border border-eng-blue/10 bg-slate-50/30 rounded-sm overflow-hidden">
                    <button
                        onClick={() => setShowBreakdown(!showBreakdown)}
                        className="w-full flex items-center justify-between p-4 bg-slate-50/80 dark:bg-slate-800/80 hover:bg-slate-100 dark:hover:bg-slate-700 transition-colors"
                    >
                        <div className="flex items-center gap-2">
                            <Calculator className="w-4 h-4 text-eng-blue" />
                            <span className="font-bold text-sm uppercase tracking-wide text-eng-navy dark:text-slate-200">Manual Calculation Breakdown</span>
                        </div>
                        {showBreakdown ? <ChevronUp className="w-4 h-4 text-slate-500" /> : <ChevronDown className="w-4 h-4 text-slate-500" />}
                    </button>

                    {showBreakdown && (
                        <div className="p-5 border-t border-slate-100 dark:border-slate-800 bg-white dark:bg-slate-900/50">
                            <div className="space-y-4">
                                {results.steps.map((step, idx) => (
                                    <div key={idx} className="border-b border-slate-100 dark:border-slate-800 last:border-0 pb-6 last:pb-0">
                                        <div className="flex justify-between items-start mb-2">
                                            <h4 className="font-bold text-xs text-eng-navy dark:text-white uppercase tracking-wider opacity-70">{step.label}</h4>
                                            <span className="font-mono text-sm font-bold text-eng-blue bg-eng-blue/5 dark:bg-eng-blue/10 px-2 py-0.5 rounded border border-eng-blue/10">
                                                {step.value}
                                            </span>
                                        </div>
                                        <div className="bg-slate-50 dark:bg-slate-950 p-4 rounded-sm border border-slate-200 dark:border-slate-800 text-xs text-slate-600 dark:text-slate-400 font-mono leading-relaxed space-y-1">
                                            {step.details.map((line, i) => (
                                                <div key={i} className={line.startsWith('=') ? 'pl-4 font-bold text-slate-700 dark:text-slate-300' : ''}>
                                                    {line}
                                                </div>
                                            ))}
                                        </div>
                                    </div>
                                ))}
                            </div>
                        </div>
                    )}
                </div>

            </div>
        </div>
    );
};
