import React, { useState, useMemo } from 'react';
import { Calculator, Box, Cuboid, Layers, Settings, Clipboard, Info } from 'lucide-react';

// --- Types ---

type Unit = 'metric' | 'imperial';
type MemberType = 'beam' | 'column' | 'slab' | 'block';
type ColumnShape = 'rectangular' | 'circular' | 'hexagonal';
type CalcMethod = 'dimension' | 'volume';
type MixRatio = '1:1:2' | '1:1.5:3' | '1:2:4' | '1:3:6' | '1:4:8' | 'custom';

// --- Constants ---

const PRESETS = {
    '1:1:2': { label: 'M25 (1:1:2)', c: 1, s: 1, a: 2 },
    '1:1.5:3': { label: 'M20 (1:1.5:3)', c: 1, s: 1.5, a: 3 },
    '1:2:4': { label: 'M15 (1:2:4)', c: 1, s: 2, a: 4 },
    '1:3:6': { label: 'M10 (1:3:6)', c: 1, s: 3, a: 6 },
    '1:4:8': { label: 'M7.5 (1:4:8)', c: 1, s: 4, a: 8 },
};

// --- Components ---

const Card: React.FC<{ title: string; icon: React.ElementType; children: React.ReactNode; className?: string }> = ({ title, icon: Icon, children, className = "" }) => (
    <div className={`bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-sm shadow-sm overflow-hidden ${className}`}>
        <div className="bg-slate-50 dark:bg-slate-800/50 px-4 py-3 border-b border-slate-200 dark:border-slate-800 flex items-center gap-2">
            <Icon className="w-4 h-4 text-eng-blue" />
            <h3 className="font-bold text-sm uppercase tracking-wider text-eng-navy dark:text-slate-100">{title}</h3>
        </div>
        <div className="p-4">
            {children}
        </div>
    </div>
);

const InputGroup: React.FC<{ label: string; children: React.ReactNode }> = ({ label, children }) => (
    <div className="space-y-1.5">
        <label className="block text-xs font-bold text-slate-500 dark:text-slate-400 uppercase tracking-wide">{label}</label>
        {children}
    </div>
);

export const ConcreteCalculator: React.FC = () => {
    // State
    const [unit, setUnit] = useState<Unit>('metric');
    const [calcMethod, setCalcMethod] = useState<CalcMethod>('dimension');
    const [memberType, setMemberType] = useState<MemberType>('column');
    const [columnShape, setColumnShape] = useState<ColumnShape>('rectangular');

    // Dimensions
    const [length, setLength] = useState<string>('0');
    const [width, setWidth] = useState<string>('0');
    const [diameter, setDiameter] = useState<string>('0');
    const [depth, setDepth] = useState<string>('0'); // Height or Depth
    const [totalVolume, setTotalVolume] = useState<string>('0');
    const [quantity, setQuantity] = useState<string>('1');

    // Tech Specs
    const [mixSelection, setMixSelection] = useState<MixRatio | 'custom'>('1:1.5:3');
    const [customMix, setCustomMix] = useState({ c: '1', s: '2', a: '4' });
    const [dryVolFactor, setDryVolFactor] = useState<string>('1.54');
    const [waterCementRatio, setWaterCementRatio] = useState<string>('0.5');
    const [tipperCapacity, setTipperCapacity] = useState<string>('10'); // Fixed m3
    const [wastage, setWastage] = useState<string>('0');

    // --- Calculation ---
    const result = useMemo(() => {
        const parseSafe = (val: string) => parseFloat(val) || 0;

        const lengthNum = parseSafe(length);
        const widthNum = parseSafe(width);
        const diameterNum = parseSafe(diameter);
        const depthNum = parseSafe(depth);
        const totalVolumeNum = parseSafe(totalVolume);
        const quantityNum = parseSafe(quantity);
        const dryVolFactorNum = parseSafe(dryVolFactor);
        const waterCementRatioNum = parseSafe(waterCementRatio);
        const wastageNum = parseSafe(wastage);
        const tipperCapacityNum = parseSafe(tipperCapacity);

        // 1. Normalize to Meters
        // Convert inputs to meters
        const toMeters = (val: number) => unit === 'imperial' ? val * 0.3048 : val;

        // For Volume: if imperial (cft), convert to m3. 1 ft3 = 0.0283168 m3
        const volToMeters = (val: number) => unit === 'imperial' ? val * 0.0283168 : val;

        let wetVol_m3 = 0;

        if (calcMethod === 'volume') {
            wetVol_m3 = volToMeters(totalVolumeNum) * quantityNum;
        } else {
            let area_m2 = 0;
            let D_m = toMeters(depthNum);

            const L_m = toMeters(lengthNum);
            const W_m = toMeters(widthNum);
            const Dia_m = toMeters(diameterNum);

            // Calculate Cross-Sectional Area based on Shape
            if (memberType === 'column') {
                if (columnShape === 'circular') {
                    const radius = Dia_m / 2;
                    area_m2 = Math.PI * Math.pow(radius, 2);
                } else if (columnShape === 'hexagonal') {
                    // Using Length as Side Length (a)
                    // Area = (3 * sqrt(3) / 2) * a^2
                    area_m2 = ((3 * Math.sqrt(3)) / 2) * Math.pow(L_m, 2);
                } else {
                    // Rectangular / Square
                    area_m2 = L_m * W_m;
                }
            } else {
                // Beam, Slab, Block are treated as Rectangular Prism for now
                area_m2 = L_m * W_m;
            }

            // 2. Volume
            wetVol_m3 = (area_m2 * D_m) * quantityNum;
        }
        const dryVol_m3 = wetVol_m3 * dryVolFactorNum * (1 + wastageNum / 100);

        if (wetVol_m3 <= 0) return null;

        // 3. Ratio
        let c = 1, s = 2, a = 4;
        if (mixSelection === 'custom') {
            c = parseSafe(customMix.c);
            s = parseSafe(customMix.s);
            a = parseSafe(customMix.a);
        } else {
            const preset = PRESETS[mixSelection as keyof typeof PRESETS];
            if (preset) {
                c = preset.c;
                s = preset.s;
                a = preset.a;
            }
        }
        const totalRatio = c + s + a;

        // 4. Materials
        const cementVol = (dryVol_m3 * c) / totalRatio;
        const cementBags = cementVol * 1440 / 50; // Density 1440 kg/m3, 50kg bag

        const sandVol = (dryVol_m3 * s) / totalRatio;
        const aggVol = (dryVol_m3 * a) / totalRatio;

        const waterLitersNum = (cementBags * 50) * waterCementRatioNum;

        // Transport Estimation
        // Capacity is ALWAYS in m3 now per user request

        const sandTripsTipper = sandVol / tipperCapacityNum;
        const aggTripsTipper = aggVol / tipperCapacityNum;

        return {
            wetVol: wetVol_m3,
            dryVol: dryVol_m3,
            cement: cementBags,
            sand: sandVol,
            aggregate: aggVol,
            water: waterLitersNum,
            sandCF: sandVol * 35.3147,
            aggCF: aggVol * 35.3147,
            transport: {
                sand: { tipper: sandTripsTipper },
                agg: { tipper: aggTripsTipper }
            }
        };
    }, [unit, calcMethod, memberType, columnShape, length, width, diameter, depth, totalVolume, quantity, mixSelection, customMix, dryVolFactor, waterCementRatio, wastage, tipperCapacity]);

    return (
        <div className="grid lg:grid-cols-2 gap-8 items-start">

            {/* Controls */}
            <div className="space-y-6">

                {/* Header Controls */}
                {/* Header Controls */}
                <div className="flex justify-between items-center gap-4">
                    <div className="flex bg-slate-100 dark:bg-slate-800 p-1 rounded-sm">
                        <button onClick={() => setUnit('imperial')} className={`px-4 py-1.5 text-xs font-bold uppercase rounded-sm transition-all ${unit === 'imperial' ? 'bg-white dark:bg-slate-600 shadow-sm text-eng-navy dark:text-white' : 'text-slate-500'}`}>Imperial (Ft)</button>
                        <button onClick={() => setUnit('metric')} className={`px-4 py-1.5 text-xs font-bold uppercase rounded-sm transition-all ${unit === 'metric' ? 'bg-white dark:bg-slate-600 shadow-sm text-eng-navy dark:text-white' : 'text-slate-500'}`}>Metric (m)</button>
                    </div>

                    <div className="flex bg-slate-100 dark:bg-slate-800 p-1 rounded-sm">
                        <button onClick={() => setCalcMethod('dimension')} className={`px-3 py-1.5 text-[10px] font-bold uppercase rounded-sm transition-all ${calcMethod === 'dimension' ? 'bg-white dark:bg-slate-600 shadow-sm text-eng-navy dark:text-white' : 'text-slate-500'}`}>By Dimension</button>
                        <button onClick={() => setCalcMethod('volume')} className={`px-3 py-1.5 text-[10px] font-bold uppercase rounded-sm transition-all ${calcMethod === 'volume' ? 'bg-white dark:bg-slate-600 shadow-sm text-eng-navy dark:text-white' : 'text-slate-500'}`}>By Volume</button>
                    </div>
                </div>

                <Card title={calcMethod === 'dimension' ? "Member Dimensions" : "Total Volume Input"} icon={Cuboid}>


                    {calcMethod === 'dimension' ? (
                        <>
                            {/* Member Type Tabs */}
                            <div className="flex border-b border-slate-200 dark:border-slate-700 mb-6">
                                {(['column', 'beam', 'slab', 'block'] as MemberType[]).map(t => (
                                    <button key={t} onClick={() => setMemberType(t)} className={`flex-1 pb-2 text-xs font-bold uppercase tracking-wide transition-colors ${memberType === t ? 'text-eng-blue border-b-2 border-eng-blue' : 'text-slate-400 hover:text-slate-600'}`}>
                                        {t}
                                    </button>
                                ))}
                            </div>

                            {/* Column Shape Selection */}
                            {memberType === 'column' && (
                                <div className="flex gap-2 mb-4 p-1 bg-slate-50 dark:bg-slate-800 rounded-sm">
                                    {(['rectangular', 'circular', 'hexagonal'] as ColumnShape[]).map(s => (
                                        <button
                                            key={s}
                                            onClick={() => setColumnShape(s)}
                                            className={`flex-1 py-1 text-[10px] font-bold uppercase rounded-sm transition-all ${columnShape === s ? 'bg-white dark:bg-slate-700 shadow-sm text-eng-blue' : 'text-slate-400'}`}
                                        >
                                            {s}
                                        </button>
                                    ))}
                                </div>
                            )}

                            <div className="grid grid-cols-2 gap-4">
                                {/* Dynamic Inputs based on Shape */}

                                {(memberType !== 'column' || columnShape === 'rectangular') && (
                                    <>
                                        <InputGroup label={`Length (${unit === 'imperial' ? 'ft' : 'm'})`}>
                                            <input type="text" value={length} onChange={e => setLength(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" placeholder="0" />
                                        </InputGroup>
                                        <InputGroup label={`Width (${unit === 'imperial' ? 'ft' : 'm'})`}>
                                            <input type="text" value={width} onChange={e => setWidth(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" placeholder="0" />
                                        </InputGroup>
                                    </>
                                )}

                                {memberType === 'column' && columnShape === 'circular' && (
                                    <div className="col-span-2">
                                        <InputGroup label={`Diameter (${unit === 'imperial' ? 'ft' : 'm'})`}>
                                            <input type="text" value={diameter} onChange={e => setDiameter(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" placeholder="Diameter" />
                                        </InputGroup>
                                    </div>
                                )}

                                {memberType === 'column' && columnShape === 'hexagonal' && (
                                    <div className="col-span-2">
                                        <InputGroup label={`Side Length (${unit === 'imperial' ? 'ft' : 'm'})`}>
                                            <input type="text" value={length} onChange={e => setLength(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" placeholder="Length of one side" />
                                        </InputGroup>
                                    </div>
                                )}

                                <InputGroup label={`${memberType === 'slab' ? 'Thickness' : memberType === 'column' ? 'Height' : 'Depth'} (${unit === 'imperial' ? 'ft' : 'm'})`}>
                                    <input type="text" value={depth} onChange={e => setDepth(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" placeholder="0" />
                                </InputGroup>
                                <InputGroup label="Quantity (Nos)">
                                    <input type="text" value={quantity} onChange={e => setQuantity(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" placeholder="1" />
                                </InputGroup>
                            </div>
                        </>
                    ) : (
                        <div className="grid grid-cols-2 gap-4">
                            <div className="col-span-2">
                                <InputGroup label={`Total Volume (${unit === 'imperial' ? 'cu.ft' : 'm³'})`}>
                                    <input type="text" value={totalVolume} onChange={e => setTotalVolume(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" placeholder="Total Volume" />
                                </InputGroup>
                            </div>
                            <InputGroup label="Quantity (Multiplier)">
                                <input type="text" value={quantity} onChange={e => setQuantity(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" placeholder="1" />
                            </InputGroup>
                        </div>
                    )}
                </Card>

                <Card title="Concrete Mix Design" icon={Settings} className="border-eng-blue/20">
                    <div className="space-y-4">
                        <InputGroup label="Grade of Concrete">
                            <select value={mixSelection} onChange={e => setMixSelection(e.target.value as any)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm text-sm">
                                {Object.entries(PRESETS).map(([k, v]) => (
                                    <option key={k} value={k}>{v.label}</option>
                                ))}
                                <option value="custom">Custom Mix Ratio</option>
                            </select>
                        </InputGroup>

                        {mixSelection === 'custom' && (
                            <div className="flex gap-2 items-center">
                                <input type="text" value={customMix.c} onChange={e => setCustomMix({ ...customMix, c: e.target.value })} className="w-16 p-2 text-center border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm text-sm" placeholder="C" />
                                <span className="font-bold text-slate-400">:</span>
                                <input type="text" value={customMix.s} onChange={e => setCustomMix({ ...customMix, s: e.target.value })} className="w-16 p-2 text-center border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm text-sm" placeholder="S" />
                                <span className="font-bold text-slate-400">:</span>
                                <input type="text" value={customMix.a} onChange={e => setCustomMix({ ...customMix, a: e.target.value })} className="w-16 p-2 text-center border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm text-sm" placeholder="A" />
                            </div>
                        )}

                        <div className="grid grid-cols-2 gap-4">
                            <InputGroup label="Dry Volume Factor">
                                <div className="relative">
                                    <input type="text" value={dryVolFactor} onChange={e => setDryVolFactor(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" placeholder="1.54" />
                                    <Info className="w-3 h-3 absolute right-2 top-3 text-slate-400" />
                                </div>
                            </InputGroup>
                            <InputGroup label="W/C Ratio">
                                <input type="text" value={waterCementRatio} onChange={e => setWaterCementRatio(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" placeholder="0.5" />
                            </InputGroup>
                            <InputGroup label="Wastage %">
                                <input type="text" value={wastage} onChange={e => setWastage(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" placeholder="0" />
                            </InputGroup>
                            {/* Placeholder to balance grid if needed */}
                        </div>

                        {/* Transport Settings */}
                        <div className="pt-4 border-t border-slate-200 dark:border-slate-700">
                            <h4 className="text-xs font-bold text-eng-navy dark:text-white uppercase tracking-wider mb-3">Transport Capacity</h4>
                            <div className="grid grid-cols-1 gap-4">
                                <InputGroup label="Tipper Capacity (m³)">
                                    <input type="text" value={tipperCapacity} onChange={e => setTipperCapacity(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" placeholder="10" />
                                </InputGroup>
                            </div>
                        </div>
                    </div>
                </Card>
            </div>

            {/* Results */}
            <div className="space-y-6">



                <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-sm shadow-sm">
                    <div className="bg-eng-navy px-4 py-3 flex justify-between items-center">
                        <h3 className="font-bold text-sm uppercase tracking-wide text-white flex items-center gap-2">
                            <Clipboard className="w-4 h-4" /> Material Estimate
                        </h3>
                    </div>

                    {result ? (
                        <div className="p-0">
                            {/* Summary */}
                            <div className="p-6 bg-slate-50 dark:bg-slate-900/50 grid grid-cols-2 gap-4 border-b border-slate-100 dark:border-slate-800">
                                <div>
                                    <div className="text-[10px] text-slate-400 uppercase font-bold tracking-wider mb-1">Total Web Volume</div>
                                    <div className="text-xl font-mono font-bold text-eng-navy dark:text-white">{result.wetVol.toFixed(3)} m³</div>
                                </div>
                                <div>
                                    <div className="text-[10px] text-slate-400 uppercase font-bold tracking-wider mb-1">Required Dry Vol</div>
                                    <div className="text-xl font-mono font-bold text-eng-blue">{result.dryVol.toFixed(3)} m³</div>
                                </div>
                            </div>

                            {/* Materials List */}
                            <div className="p-6 space-y-6">
                                {/* Cement */}
                                <div className="flex justify-between items-center">
                                    <div className="flex items-center gap-3">
                                        <div className="w-10 h-10 rounded-full bg-gray-200 dark:bg-gray-800 flex items-center justify-center text-gray-600 font-bold">C</div>
                                        <div>
                                            <div className="font-bold text-eng-navy dark:text-white">Cement</div>
                                            <div className="text-xs text-slate-400">50kg Bags</div>
                                        </div>
                                    </div>
                                    <div className="text-right">
                                        <div className="text-2xl font-mono font-bold text-eng-navy dark:text-white">{Math.ceil(result.cement)} <span className="text-sm font-sans text-slate-400">bags</span></div>
                                        <div className="text-[10px] text-slate-400">({result.cement.toFixed(2)} exact)</div>
                                    </div>
                                </div>

                                {/* Sand */}
                                <div className="flex justify-between items-center">
                                    <div className="flex items-center gap-3">
                                        <div className="w-10 h-10 rounded-full bg-yellow-100 dark:bg-yellow-900/20 flex items-center justify-center text-yellow-600 font-bold">S</div>
                                        <div>
                                            <div className="font-bold text-eng-navy dark:text-white">Sand</div>
                                            <div className="text-xs text-slate-400">Fine Aggregate</div>
                                        </div>
                                    </div>
                                    <div className="text-right">
                                        <div className="text-2xl font-mono font-bold text-eng-navy dark:text-white">{result.sand.toFixed(3)} <span className="text-sm font-sans text-slate-400">m³</span></div>
                                        <div className="text-[10px] text-slate-400 mb-1">{result.sandCF.toFixed(2)} cu.ft</div>
                                        <div className="text-[10px] text-eng-blue font-bold bg-blue-50 dark:bg-blue-900/20 px-1.5 py-0.5 rounded-sm inline-block">
                                            {result.transport.sand.tipper.toFixed(1)} Tipper Trips
                                        </div>
                                    </div>
                                </div>

                                {/* Aggregate */}
                                <div className="flex justify-between items-center">
                                    <div className="flex items-center gap-3">
                                        <div className="w-10 h-10 rounded-full bg-stone-200 dark:bg-stone-800 flex items-center justify-center text-stone-600 font-bold">A</div>
                                        <div>
                                            <div className="font-bold text-eng-navy dark:text-white">Aggregate</div>
                                            <div className="text-xs text-slate-400">Coarse Aggregate</div>
                                        </div>
                                    </div>
                                    <div className="text-right">
                                        <div className="text-2xl font-mono font-bold text-eng-navy dark:text-white">{result.aggregate.toFixed(3)} <span className="text-sm font-sans text-slate-400">m³</span></div>
                                        <div className="text-[10px] text-slate-400 mb-1">{result.aggCF.toFixed(2)} cu.ft</div>
                                        <div className="text-[10px] text-eng-blue font-bold bg-blue-50 dark:bg-blue-900/20 px-1.5 py-0.5 rounded-sm inline-block">
                                            {result.transport.agg.tipper.toFixed(1)} Tipper Trips
                                        </div>
                                    </div>
                                </div>

                                {/* Water */}
                                <div className="flex justify-between items-center bg-blue-50 dark:bg-blue-900/10 p-2 rounded-sm -mx-2">
                                    <div className="flex items-center gap-3">
                                        <div className="w-10 h-10 rounded-full bg-blue-100 dark:bg-blue-900/30 flex items-center justify-center text-blue-600 font-bold">W</div>
                                        <div>
                                            <div className="font-bold text-eng-navy dark:text-white">Water</div>
                                            <div className="text-xs text-slate-400">Litres Required</div>
                                        </div>
                                    </div>
                                    <div className="text-right">
                                        <div className="text-2xl font-mono font-bold text-eng-blue">{Math.ceil(result.water)} <span className="text-sm font-sans text-slate-400">L</span></div>
                                        <div className="text-[10px] text-slate-400">@ W/C {waterCementRatio}</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    ) : (
                        <div className="p-12 text-center text-slate-400">
                            <Box className="w-12 h-12 mx-auto mb-3 opacity-20" />
                            <p>Enter dimensions to calculate material estimation.</p>
                        </div>
                    )}
                </div>
            </div>
        </div >
    );
};
