import React, { useState, useMemo, useEffect } from 'react';
import {
    Calculator, Plus, Trash2, Clipboard, Settings, Ruler, Grid, RefreshCw, Box, Layers, Droplet, ArrowRight, ArrowUp, ChevronDown, ChevronUp
} from 'lucide-react';

// --- Types ---

type Unit = 'imperial' | 'metric';
type CalcMode = 'dimensions' | 'volume';

interface Deduction {
    id: string;
    type: 'Door' | 'Window' | 'Custom';
    l: string; // m
    h: string; // m

    // UI Helpers
    l_ft?: string;
    l_in?: string;
    h_ft?: string;
    h_in?: string;

    count: string;
}

interface CalculationResult {
    totalArea: number;
    deductionArea: number;
    netArea: number;
    volume: number;
    bricks: number;
    cement: number; // Bags
    sand: number; // Cu.ft or m3
    water: number; // Liters
    mortarDryVol: number;
    steps: { label: string; value: string; formula?: string }[];
}

// --- Constants ---

const BRICK_SIZE_MM = { l: 230, w: 115, h: 57 }; // Standard Nepal Brick (User Specified)
const BRICK_SIZE_INCH = { l: 9.05512, w: 4.52756, h: 2.24409 }; // Precise conversion of 230x115x57mm

// Defaults for Mortar
const DEFAULT_MORTAR_MM = 10;
const DEFAULT_MORTAR_INCH = 0.3937; // 10mm in inches

// --- Helper Components ---

const Card: React.FC<{ title: string; icon?: React.ElementType; children: React.ReactNode; className?: string }> = ({ title, icon: Icon, children, className = "" }) => (
    <div className={`bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-sm shadow-sm overflow-hidden ${className}`}>
        <div className="bg-slate-50 dark:bg-slate-950 px-4 py-3 border-b border-slate-200 dark:border-slate-800 flex items-center gap-2">
            {Icon && <Icon className="w-4 h-4 text-eng-blue" />}
            <h3 className="font-bold text-sm uppercase tracking-wide text-eng-navy dark:text-slate-200">{title}</h3>
        </div>
        <div className="p-5">{children}</div>
    </div>
);

const InputGroup: React.FC<{ label: string; suffix?: string; children: React.ReactNode }> = ({ label, suffix, children }) => (
    <div className="mb-4 last:mb-0">
        <label className="block text-xs font-bold text-slate-500 uppercase mb-1">{label}</label>
        <div className="relative">
            {children}
            {suffix && <span className="absolute right-3 top-1/2 -translate-y-1/2 text-slate-400 text-sm font-mono">{suffix}</span>}
        </div>
    </div>
);

// --- Main Component ---

export const BrickCalculator: React.FC = () => {
    // State
    const [unit, setUnit] = useState<Unit>('metric');
    const [calcMode, setCalcMode] = useState<CalcMode>('dimensions');
    const [showBreakdown, setShowBreakdown] = useState(false);

    // Input State
    const [lengthFt, setLengthFt] = useState<string>('10');
    const [lengthIn, setLengthIn] = useState<string>('0');
    const [lengthM, setLengthM] = useState<string>('1');

    const [heightFt, setHeightFt] = useState<string>('10');
    const [heightIn, setHeightIn] = useState<string>('0');
    const [heightM, setHeightM] = useState<string>('1');

    const [directVolume, setDirectVolume] = useState<string>('1');

    // Thickness
    const [thicknessType, setThicknessType] = useState<number>(0);
    const [customThickness, setCustomThickness] = useState<string>('1000');

    const [deductions, setDeductions] = useState<Deduction[]>([]);

    // Tech Specs State
    const [dryVolPercentage, setDryVolPercentage] = useState<string>('30');
    const [wastagePercentage, setWastagePercentage] = useState<string>('5');
    const [wcRatio, setWcRatio] = useState<string>('0.5');
    const [mortarRatio, setMortarRatio] = useState("1:3");

    // Mortar Thickness State
    const [mortarThick_mm, setMortarThick_mm] = useState<string>(DEFAULT_MORTAR_MM.toString());
    const [mortarThick_in, setMortarThick_in] = useState<string>(DEFAULT_MORTAR_INCH.toString());

    // Brick Size State
    const [brickL_mm, setBrickL_mm] = useState<string>(BRICK_SIZE_MM.l.toString());
    const [brickW_mm, setBrickW_mm] = useState<string>(BRICK_SIZE_MM.w.toString());
    const [brickH_mm, setBrickH_mm] = useState<string>(BRICK_SIZE_MM.h.toString());

    const [brickL_in, setBrickL_in] = useState<string>(BRICK_SIZE_INCH.l.toString());
    const [brickW_in, setBrickW_in] = useState<string>(BRICK_SIZE_INCH.w.toString());
    const [brickH_in, setBrickH_in] = useState<string>(BRICK_SIZE_INCH.h.toString());

    // Derived Values
    const thicknessNum = thicknessType === 0 ? parseFloat(customThickness) || 0 : thicknessType;

    // Getters
    const getLength = () => unit === 'imperial' ? (parseFloat(lengthFt) || 0) + ((parseFloat(lengthIn) || 0) / 12) : (parseFloat(lengthM) || 0);
    const getHeight = () => unit === 'imperial' ? (parseFloat(heightFt) || 0) + ((parseFloat(heightIn) || 0) / 12) : (parseFloat(heightM) || 0);

    // --- Logic ---

    const addDeduction = (type: Deduction['type']) => {
        setDeductions([...deductions, {
            id: Math.random().toString(36).substr(2, 9),
            type,
            l: type === 'Door' ? '3' : '1',
            h: type === 'Door' ? '7' : '4',
            l_ft: type === 'Door' ? '3' : '1', l_in: '0',
            h_ft: type === 'Door' ? '7' : '4', h_in: '0',
            count: '1'
        }]);
    };

    const updateDeduction = (id: string, field: keyof Deduction, value: any) => {
        setDeductions(prev => prev.map(d => {
            if (d.id !== id) return d;
            const updated = { ...d, [field]: value };
            if (unit === 'imperial') {
                if (field === 'l_ft' || field === 'l_in') {
                    updated.l = ((parseFloat(updated.l_ft || '0') || 0) + ((parseFloat(updated.l_in || '0') || 0) / 12)).toString();
                }
                if (field === 'h_ft' || field === 'h_in') {
                    updated.h = ((parseFloat(updated.h_ft || '0') || 0) + ((parseFloat(updated.h_in || '0') || 0) / 12)).toString();
                }
            }
            return updated;
        }));
    };

    const removeDeduction = (id: string) => {
        setDeductions(deductions.filter(d => d.id !== id));
    };

    const result: CalculationResult | null = useMemo(() => {
        const parseSafe = (val: string) => parseFloat(val) || 0;

        let wallVol_m3 = 0;
        let totalArea_m2 = 0;
        let deductionArea_m2 = 0;
        let netArea_m2 = 0;

        const lengthVal = getLength();
        const heightVal = getHeight();

        // 1. Determine Wall Dimensions in Meters
        let L_m, H_m, WallT_m;

        if (calcMode === 'dimensions') {
            if (!lengthVal || !heightVal) return null;

            L_m = unit === 'imperial' ? lengthVal * 0.3048 : lengthVal;
            H_m = unit === 'imperial' ? heightVal * 0.3048 : heightVal;

            if (unit === 'imperial') {
                WallT_m = thicknessNum * 0.0254;
            } else {
                if (thicknessType === 4) WallT_m = 0.110;
                else if (thicknessType === 9) WallT_m = 0.230;
                else if (thicknessType === 14) WallT_m = 0.350;
                else WallT_m = parseSafe(customThickness) / 1000;
            }

            totalArea_m2 = L_m * H_m;

            deductions.forEach(d => {
                const valL = unit === 'imperial' ? ((parseSafe(d.l_ft || '0')) + (parseSafe(d.l_in || '0')) / 12) : parseSafe(d.l);
                const valH = unit === 'imperial' ? ((parseSafe(d.h_ft || '0')) + (parseSafe(d.h_in || '0')) / 12) : parseSafe(d.h);

                const dL_m = unit === 'imperial' ? valL * 0.3048 : valL;
                const dH_m = unit === 'imperial' ? valH * 0.3048 : valH;
                deductionArea_m2 += (dL_m * dH_m * parseSafe(d.count));
            });

            netArea_m2 = Math.max(0, totalArea_m2 - deductionArea_m2);
            wallVol_m3 = netArea_m2 * WallT_m;

        } else {
            const directVolNum = parseSafe(directVolume);
            if (!directVolNum) return null;
            wallVol_m3 = unit === 'imperial'
                ? directVolNum * 0.0283168
                : directVolNum;
        }

        let bL_m, bW_m, bH_m, mortar_m;

        if (unit === 'imperial') {
            bL_m = parseSafe(brickL_in) * 0.0254;
            bW_m = parseSafe(brickW_in) * 0.0254;
            bH_m = parseSafe(brickH_in) * 0.0254;
            mortar_m = parseSafe(mortarThick_in) * 0.0254;
        } else {
            bL_m = parseSafe(brickL_mm) / 1000;
            bW_m = parseSafe(brickW_mm) / 1000;
            bH_m = parseSafe(brickH_mm) / 1000;
            mortar_m = parseSafe(mortarThick_mm) / 1000;
        }

        const volOneBrickWithMortar = (bL_m + mortar_m) * (bW_m + mortar_m) * (bH_m + mortar_m);

        // Total Bricks with Wastage
        const rawBricks = wallVol_m3 / volOneBrickWithMortar;
        const numBricks = Math.ceil(rawBricks * (1 + (parseSafe(wastagePercentage) / 100)));

        // Mortar Volume
        const theoreticalBricks = Math.ceil(wallVol_m3 / volOneBrickWithMortar);
        const volBricksOnly = theoreticalBricks * (bL_m * bW_m * bH_m);
        const wetMortarVol = Math.max(0, wallVol_m3 - volBricksOnly);

        // Dynamic Dry Volume Conversion
        const dryMortarVol = wetMortarVol * (1 + (parseSafe(dryVolPercentage) / 100));

        // Materials
        const [cRatio, sRatio] = mortarRatio.split(':').map(Number);
        const totalRatio = cRatio + sRatio;

        const cementVol = (dryMortarVol * cRatio) / totalRatio;
        const cementBags = parseFloat((cementVol * 28.8).toFixed(1));

        const sandVol = (dryMortarVol * sRatio) / totalRatio;
        const sandOut = unit === 'imperial'
            ? parseFloat((sandVol * 35.3147).toFixed(1))
            : parseFloat(sandVol.toFixed(2));

        // Water Calculation
        const cementWeightKg = cementBags * 50;
        const waterLiters = cementWeightKg * parseSafe(wcRatio);

        // --- Generate Steps ---
        const steps: { label: string; value: string; formula?: string }[] = [];

        steps.push({
            label: "1. Wall Volume",
            value: `${wallVol_m3.toFixed(3)} m³`,
            formula: calcMode === 'dimensions'
                ? `Net Area (${netArea_m2.toFixed(2)} m²) × Thickness (${WallT_m?.toFixed(3)} m)`
                : `Direct Input`
        });

        steps.push({
            label: "2. Volume of 1 Brick (with Mortar)",
            value: `${volOneBrickWithMortar.toFixed(6)} m³`,
            formula: `(${bL_m.toFixed(3)} + ${mortar_m}) × (${bW_m.toFixed(3)} + ${mortar_m}) × (${bH_m.toFixed(3)} + ${mortar_m})`
        });

        steps.push({
            label: "3. Total Bricks Required",
            value: `${numBricks.toFixed(0)} Bricks`,
            formula: `(Wall Vol ${wallVol_m3.toFixed(3)} / One Brick Vol ${volOneBrickWithMortar.toFixed(6)}) + ${wastagePercentage}% Wastage`
        });

        steps.push({
            label: "4. Wet Mortar Volume",
            value: `${wetMortarVol.toFixed(4)} m³`,
            formula: `Wall Vol - (Total Bricks (no wastage) × Brick Vol (no mortar))`
        });

        steps.push({
            label: "5. Dry Mortar Volume",
            value: `${dryMortarVol.toFixed(4)} m³`,
            formula: `Wet Vol × (1 + ${(parseSafe(dryVolPercentage) / 100).toFixed(2)}) (Dry Coeff)`
        });

        steps.push({
            label: "6. Cement Quantity",
            value: `${cementBags} Bags`,
            formula: `(Dry Vol × Cement Ratio ${cRatio}/${totalRatio}) × 28.8 bags/m³`
        });

        steps.push({
            label: "7. Sand Quantity",
            value: `${sandOut} ${unit === 'imperial' ? 'cu.ft' : 'm³'}`,
            formula: unit === 'imperial'
                ? `(Dry Vol × Sand Ratio ${sRatio}/${totalRatio}) × 35.3147`
                : `(Dry Vol × Sand Ratio ${sRatio}/${totalRatio})`
        });

        return {
            totalArea: parseFloat((unit === 'imperial' ? totalArea_m2 * 10.764 : totalArea_m2).toFixed(2)),
            deductionArea: parseFloat((unit === 'imperial' ? deductionArea_m2 * 10.764 : deductionArea_m2).toFixed(2)),
            netArea: parseFloat((unit === 'imperial' ? netArea_m2 * 10.764 : netArea_m2).toFixed(2)),
            volume: parseFloat(wallVol_m3.toFixed(2)),
            bricks: numBricks,
            cement: cementBags,
            sand: sandOut,
            water: parseFloat(waterLiters.toFixed(0)),
            mortarDryVol: parseFloat(dryMortarVol.toFixed(3)),
            steps
        };

    }, [unit, calcMode,
        lengthFt, lengthIn, lengthM,
        heightFt, heightIn, heightM,
        directVolume, thicknessType, customThickness, deductions,
        brickL_mm, brickH_mm, brickW_mm, brickL_in, brickH_in, brickW_in,
        mortarRatio, dryVolPercentage, wastagePercentage, wcRatio,
        mortarThick_mm, mortarThick_in]);


    // --- Render ---

    // Visual Helper Functions
    const getDeductionLabel = (type: string, index: number) => {
        const prefix = type === 'Door' ? 'D' : type === 'Window' ? 'W' : 'O';
        return `${prefix}${index + 1}`;
    };

    return (
        <div className="grid lg:grid-cols-2 gap-8 items-start">

            {/* LEFT PANEL: CONTROLS */}
            <div className="space-y-6">

                {/* Top Controls: Unit & Mode */}
                <div className="flex flex-col gap-2">

                    {/* Unit Toggle */}
                    <div className="flex bg-slate-100 dark:bg-slate-800 p-1 rounded-sm w-fit self-start">
                        <button
                            onClick={() => setUnit('imperial')}
                            className={`px-4 py-1.5 text-xs font-bold uppercase tracking-wider rounded-sm transition-all ${unit === 'imperial' ? 'bg-white dark:bg-slate-600 shadow-sm text-eng-navy dark:text-white' : 'text-slate-500'}`}
                        >
                            Imperial (Ft/In)
                        </button>
                        <button
                            onClick={() => setUnit('metric')}
                            className={`px-4 py-1.5 text-xs font-bold uppercase tracking-wider rounded-sm transition-all ${unit === 'metric' ? 'bg-white dark:bg-slate-600 shadow-sm text-eng-navy dark:text-white' : 'text-slate-500'}`}
                        >
                            Metric (M)
                        </button>
                    </div>
                </div>

                {/* Card 1: Wall Data */}
                <Card title="Wall Data" icon={Ruler}>
                    {/* Calculation Mode Tabs */}
                    <div className="flex border-b border-slate-200 dark:border-slate-700 mb-4">
                        <button
                            onClick={() => setCalcMode('dimensions')}
                            className={`flex-1 pb-2 text-xs font-bold uppercase tracking-wide transition-colors ${calcMode === 'dimensions' ? 'text-eng-blue border-b-2 border-eng-blue' : 'text-slate-400 hover:text-slate-600'}`}
                        >
                            By Dimensions
                        </button>
                        <button
                            onClick={() => setCalcMode('volume')}
                            className={`flex-1 pb-2 text-xs font-bold uppercase tracking-wide transition-colors ${calcMode === 'volume' ? 'text-eng-blue border-b-2 border-eng-blue' : 'text-slate-400 hover:text-slate-600'}`}
                        >
                            By Volume
                        </button>
                    </div>

                    {calcMode === 'dimensions' ? (
                        <>
                            <div className="grid grid-cols-2 gap-4">
                                {/* LENGTH INPUT */}
                                <div>
                                    <label className="block text-xs font-bold text-slate-500 uppercase mb-1">Length</label>
                                    {unit === 'imperial' ? (
                                        <div className="flex gap-2">
                                            <div className="relative flex-1">
                                                <input type="text" value={lengthFt} onChange={e => setLengthFt(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" placeholder="Ft" />
                                                <span className="absolute right-2 top-1.5 text-xs text-slate-400 pointer-events-none">ft</span>
                                            </div>
                                            <div className="relative w-24">
                                                <input type="text" value={lengthIn} onChange={e => setLengthIn(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" placeholder="In" />
                                                <span className="absolute right-2 top-1.5 text-xs text-slate-400 pointer-events-none">in</span>
                                            </div>
                                        </div>
                                    ) : (
                                        <div className="relative">
                                            <input type="text" value={lengthM} onChange={e => setLengthM(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" />
                                            <span className="absolute right-3 top-1.5 text-xs text-slate-400 pointer-events-none">m</span>
                                        </div>
                                    )}
                                </div>

                                {/* HEIGHT INPUT */}
                                <div>
                                    <label className="block text-xs font-bold text-slate-500 uppercase mb-1">Height</label>
                                    {unit === 'imperial' ? (
                                        <div className="flex gap-2">
                                            <div className="relative flex-1">
                                                <input type="text" value={heightFt} onChange={e => setHeightFt(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" placeholder="Ft" />
                                                <span className="absolute right-2 top-1.5 text-xs text-slate-400 pointer-events-none">ft</span>
                                            </div>
                                            <div className="relative w-24">
                                                <input type="text" value={heightIn} onChange={e => setHeightIn(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" placeholder="In" />
                                                <span className="absolute right-2 top-1.5 text-xs text-slate-400 pointer-events-none">in</span>
                                            </div>
                                        </div>
                                    ) : (
                                        <div className="relative">
                                            <input type="text" value={heightM} onChange={e => setHeightM(e.target.value)} className="w-full p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm" />
                                            <span className="absolute right-3 top-1.5 text-xs text-slate-400 pointer-events-none">m</span>
                                        </div>
                                    )}
                                </div>
                            </div>

                            <InputGroup label="Wall Thickness">
                                <div className="flex gap-2">
                                    <select
                                        value={thicknessType}
                                        onChange={e => {
                                            const val = parseFloat(e.target.value);
                                            setThicknessType(val);
                                            // Keep customThickness as string
                                            if (val !== 0) setCustomThickness(val.toString());
                                        }}
                                        className="flex-1 p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm text-sm"
                                    >
                                        <option value={0}>Custom / Manual</option>
                                        <option value={4}>{unit === 'imperial' ? '4 inch (Half Brick)' : '110mm (Half Brick)'}</option>
                                        <option value={9}>{unit === 'imperial' ? '9 inch (Full Brick)' : '230mm (Full Brick)'}</option>
                                        <option value={14}>{unit === 'imperial' ? '14 inch (One & Half)' : '350mm (One & Half)'}</option>
                                    </select>
                                    {thicknessType === 0 && (
                                        <input
                                            type="text"
                                            placeholder={unit === 'imperial' ? 'Inch' : 'mm'}
                                            value={customThickness}
                                            onChange={e => setCustomThickness(e.target.value)}
                                            className="w-24 p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-sm border-l-4 border-l-eng-blue"
                                        />
                                    )}
                                </div>
                            </InputGroup>
                        </>
                    ) : (
                        <div className="py-2">
                            <InputGroup label={`Total Wall Volume (${unit === 'imperial' ? 'Cu.Ft' : 'm³'})`}>
                                <div className="flex items-center">
                                    <Box className="absolute left-3 w-4 h-4 text-slate-400" />
                                    <input
                                        type="text"
                                        value={directVolume}
                                        onChange={e => setDirectVolume(e.target.value)}
                                        className="w-full pl-10 p-2 border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-lg font-bold text-eng-navy dark:text-white"
                                        placeholder="Enter Volume"
                                    />
                                </div>
                                <p className="text-[10px] text-slate-400 mt-1 italic">* Enter the net volume of brickwork directly.</p>
                            </InputGroup>
                        </div>
                    )}
                </Card>

                {/* Card 2: Technical Specifications (ALWAYS VISIBLE) */}
                <Card title="Technical Specifications" icon={Settings} className="border-eng-blue/20">
                    <div className="space-y-4">
                        <div className="grid grid-cols-2 gap-4">
                            <InputGroup label={`Brick Size (${unit === 'imperial' ? 'inch' : 'mm'})`}>
                                <div className="flex items-center gap-1">
                                    {unit === 'imperial' ? (
                                        <>
                                            <input value={brickL_in} onChange={e => setBrickL_in(e.target.value)} className="w-full p-1 text-xs border dark:border-slate-700 bg-white dark:bg-slate-800 font-mono rounded-sm" placeholder="L" />
                                            <span className="text-slate-300">x</span>
                                            <input value={brickW_in} onChange={e => setBrickW_in(e.target.value)} className="w-full p-1 text-xs border dark:border-slate-700 bg-white dark:bg-slate-800 font-mono rounded-sm" placeholder="W" />
                                            <span className="text-slate-300">x</span>
                                            <input value={brickH_in} onChange={e => setBrickH_in(e.target.value)} className="w-full p-1 text-xs border dark:border-slate-700 bg-white dark:bg-slate-800 font-mono rounded-sm" placeholder="H" />
                                        </>
                                    ) : (
                                        <>
                                            <input value={brickL_mm} onChange={e => setBrickL_mm(e.target.value)} className="w-full p-1 text-xs border dark:border-slate-700 bg-white dark:bg-slate-800 font-mono rounded-sm" placeholder="L" />
                                            <span className="text-slate-300">x</span>
                                            <input value={brickW_mm} onChange={e => setBrickW_mm(e.target.value)} className="w-full p-1 text-xs border dark:border-slate-700 bg-white dark:bg-slate-800 font-mono rounded-sm" placeholder="W" />
                                            <span className="text-slate-300">x</span>
                                            <input value={brickH_mm} onChange={e => setBrickH_mm(e.target.value)} className="w-full p-1 text-xs border dark:border-slate-700 bg-white dark:bg-slate-800 font-mono rounded-sm" placeholder="H" />
                                        </>
                                    )}
                                </div>
                            </InputGroup>

                            {/* Mortar Thickness Input */}
                            <InputGroup label={`Mortar Thick. (${unit === 'imperial' ? 'inch' : 'mm'})`}>
                                {unit === 'imperial' ? (
                                    <input
                                        type="text"
                                        value={mortarThick_in}
                                        onChange={e => setMortarThick_in(e.target.value)}
                                        className="w-full p-2 text-xs border dark:border-slate-700 bg-white dark:bg-slate-800 font-mono rounded-sm"
                                        placeholder="0.39"
                                    />
                                ) : (
                                    <input
                                        type="text"
                                        value={mortarThick_mm}
                                        onChange={e => setMortarThick_mm(e.target.value)}
                                        className="w-full p-2 text-xs border dark:border-slate-700 bg-white dark:bg-slate-800 font-mono rounded-sm"
                                        placeholder="10"
                                    />
                                )}
                            </InputGroup>
                        </div>

                        <div className="grid grid-cols-2 gap-4">
                            <InputGroup label="Brick Wastage (%)">
                                <input
                                    type="text"
                                    value={wastagePercentage}
                                    onChange={e => setWastagePercentage(e.target.value)}
                                    className="w-full p-2 text-xs border font-mono rounded-sm bg-red-50 dark:bg-red-900/10 border-red-100 dark:border-red-900/30"
                                />
                            </InputGroup>
                            <InputGroup label="Dry Volume Incr. (%)">
                                <input
                                    type="text"
                                    value={dryVolPercentage}
                                    onChange={e => setDryVolPercentage(e.target.value)}
                                    className="w-full p-2 text-xs border dark:border-slate-700 bg-white dark:bg-slate-800 font-mono rounded-sm"
                                />
                            </InputGroup>
                        </div>

                        <div className="grid grid-cols-2 gap-4">
                            <InputGroup label="Mortar Ratio">
                                <select value={mortarRatio} onChange={e => setMortarRatio(e.target.value)} className="w-full p-2 text-xs border font-mono rounded-sm">
                                    <option value="1:3">1:3 (Rich)</option>
                                    <option value="1:4">1:4 (External)</option>
                                    <option value="1:5">1:5 (Internal)</option>
                                    <option value="1:6">1:6 (Lean)</option>
                                </select>
                            </InputGroup>
                            <InputGroup label="W/C Ratio">
                                <input
                                    type="text"
                                    value={wcRatio}
                                    onChange={e => setWcRatio(e.target.value)}
                                    className="w-full p-2 text-xs border font-mono rounded-sm bg-blue-50 dark:bg-blue-900/10 border-blue-100 dark:border-blue-900/30"
                                />
                            </InputGroup>
                        </div>
                    </div>
                </Card>

                {/* Card 3: Deductions (Only in Dimensions Mode) */}
                {calcMode === 'dimensions' && (
                    <Card title="Deductions (Openings)" icon={Grid}>
                        <div className="space-y-3 mb-4">
                            {deductions.map((d, i) => (
                                <div key={d.id} className="flex gap-2 items-center bg-slate-50 dark:bg-slate-900/50 p-2 rounded-sm border border-slate-100 dark:border-slate-800">
                                    <span className="text-xs font-bold w-12 text-slate-500">{getDeductionLabel(d.type, i)}</span>

                                    {unit === 'imperial' ? (
                                        <>
                                            {/* FT */}
                                            <div className="relative">
                                                <input type="text"
                                                    value={d.l_ft} onChange={e => updateDeduction(d.id, 'l_ft', e.target.value)}
                                                    className="w-12 pl-2 pr-4 py-1 text-xs border border-slate-300 rounded-sm font-mono" placeholder="Ft" />
                                                <span className="absolute right-1 top-1/2 -translate-y-1/2 text-[9px] text-slate-400">ft</span>
                                            </div>

                                            {/* IN */}
                                            <div className="relative">
                                                <input type="text"
                                                    value={d.l_in} onChange={e => updateDeduction(d.id, 'l_in', e.target.value)}
                                                    className="w-10 pl-2 pr-4 py-1 text-xs border border-slate-300 rounded-sm font-mono" placeholder="In" />
                                                <span className="absolute right-1 top-1/2 -translate-y-1/2 text-[9px] text-slate-400">in</span>
                                            </div>

                                            <span className="text-slate-400 text-[10px]">x</span>

                                            {/* FT */}
                                            <div className="relative">
                                                <input type="text"
                                                    value={d.h_ft} onChange={e => updateDeduction(d.id, 'h_ft', e.target.value)}
                                                    className="w-12 pl-2 pr-4 py-1 text-xs border border-slate-300 rounded-sm font-mono" placeholder="Ft" />
                                                <span className="absolute right-1 top-1/2 -translate-y-1/2 text-[9px] text-slate-400">ft</span>
                                            </div>

                                            {/* IN */}
                                            <div className="relative">
                                                <input type="text"
                                                    value={d.h_in} onChange={e => updateDeduction(d.id, 'h_in', e.target.value)}
                                                    className="w-10 pl-2 pr-4 py-1 text-xs border border-slate-300 rounded-sm font-mono" placeholder="In" />
                                                <span className="absolute right-1 top-1/2 -translate-y-1/2 text-[9px] text-slate-400">in</span>
                                            </div>
                                        </>
                                    ) : (
                                        <>
                                            <div className="relative">
                                                <input
                                                    type="text"
                                                    value={d.l}
                                                    onChange={e => updateDeduction(d.id, 'l', e.target.value)}
                                                    className="w-16 pl-2 pr-4 py-1 text-xs border border-slate-300 rounded-sm font-mono"
                                                    placeholder="L"
                                                />
                                                <span className="absolute right-1 top-1/2 -translate-y-1/2 text-[9px] text-slate-400">m</span>
                                            </div>

                                            <span className="text-slate-400 text-[10px]">x</span>

                                            <div className="relative">
                                                <input
                                                    type="text"
                                                    value={d.h}
                                                    onChange={e => updateDeduction(d.id, 'h', e.target.value)}
                                                    className="w-16 pl-2 pr-4 py-1 text-xs border border-slate-300 rounded-sm font-mono"
                                                    placeholder="H"
                                                />
                                                <span className="absolute right-1 top-1/2 -translate-y-1/2 text-[9px] text-slate-400">m</span>
                                            </div>
                                        </>
                                    )}

                                    <span className="text-slate-400 text-[10px]">@</span>
                                    <input
                                        type="text"
                                        value={d.count}
                                        onChange={e => updateDeduction(d.id, 'count', e.target.value)}
                                        className="w-10 p-1 text-xs border border-slate-300 dark:border-slate-700 bg-white dark:bg-slate-800 rounded-sm font-mono text-center"
                                    />
                                    <button onClick={() => removeDeduction(d.id)} className="ml-auto text-red-400 hover:text-red-600">
                                        <Trash2 className="w-4 h-4" />
                                    </button>
                                </div>
                            ))}
                            {deductions.length === 0 && <p className="text-xs text-slate-400 italic text-center py-2">No deductions (Solid Wall)</p>}
                        </div>

                        <div className="flex gap-2">
                            <button onClick={() => addDeduction('Door')} className="flex-1 py-2 text-xs font-bold border border-dashed border-slate-300 text-slate-600 hover:bg-slate-50 hover:border-slate-400 rounded-sm transition-colors flex items-center justify-center gap-1">
                                <Plus className="w-3 h-3" /> Add Door
                            </button>
                            <button onClick={() => addDeduction('Window')} className="flex-1 py-2 text-xs font-bold border border-dashed border-slate-300 text-slate-600 hover:bg-slate-50 hover:border-slate-400 rounded-sm transition-colors flex items-center justify-center gap-1">
                                <Plus className="w-3 h-3" /> Add Window
                            </button>
                        </div>
                    </Card>
                )}

            </div>

            {/* RIGHT PANEL: BLUEPRINT & RESULTS */}
            <div className="space-y-6">

                {/* Visual Preview */}
                <div className="bg-eng-navy p-1 rounded-sm shadow-md">
                    <div className="bg-[#f0f4f8] relative h-[300px] w-full rounded-sm overflow-hidden flex items-center justify-center">
                        {/* Blueprint Label */}
                        <div className="absolute top-3 left-3 px-2 py-1 bg-white/80 backdrop-blur-sm border border-slate-200 text-[10px] font-bold uppercase tracking-widest text-slate-500 rounded-sm z-10">
                            {calcMode === 'dimensions' ? 'Wall Elevation' : 'Direct Volume Mode'}
                        </div>

                        {calcMode === 'dimensions' ? (
                            <>
                                {/* SVG Drawing */}
                                {getLength() > 0 && getHeight() > 0 && (
                                    <div className="relative p-4 w-full h-full flex items-center justify-center">
                                        {/* 
                           Proportional ViewBox for consistent sizing
                        */}
                                        {(() => {
                                            const L = getLength();
                                            const H = getHeight();
                                            const padX = L * 0.15;
                                            const padY = H * 0.15;
                                            const viewBoxStr = `${-padX} ${-padY} ${L + padX * 2} ${H + padY * 2}`;
                                            // Brick Pattern Size: 230mm (0.23m) or 9 inch (0.75ft)
                                            const brickW = unit === 'imperial' ? 0.75 : 0.23;
                                            const brickH = unit === 'imperial' ? 0.25 : 0.076;

                                            return (
                                                <svg
                                                    width="100%"
                                                    height="100%"
                                                    viewBox={viewBoxStr}
                                                    preserveAspectRatio="xMidYMid meet"
                                                    className="max-w-full max-h-full"
                                                    style={{ filter: "drop-shadow(0px 4px 6px rgba(0,0,0,0.1))" }}
                                                >
                                                    <defs>
                                                        <pattern id="brickPattern" patternUnits="userSpaceOnUse" width={brickW} height={brickH} viewBox={`0 0 ${brickW * 10} ${brickH * 10}`}>
                                                            <rect width={brickW * 10} height={brickH * 10} fill="#f1f5f9" />
                                                            {/* Scaled Brick Lines */}
                                                            <path d={`M0 ${brickH * 10} L${brickW * 10} ${brickH * 10} M${brickW * 5} 0 L${brickW * 5} ${brickH * 10}`} stroke="#cbd5e1" strokeWidth={Math.min(L, H) * 0.002} fill="none" />
                                                        </pattern>
                                                        {/* Dimension Arrow Marker */}
                                                        <marker id="arrow" markerWidth="6" markerHeight="6" refX="3" refY="3" orient="auto">
                                                            <path d="M0,0 L6,3 L0,6 L1.5,3 Z" fill="#64748b" />
                                                        </marker>
                                                    </defs>

                                                    {/* Wall Body */}
                                                    <rect x="0" y="0" width={L} height={H} fill="url(#brickPattern)" stroke="#475569" strokeWidth={Math.min(L, H) * 0.01} />

                                                    {/* Deductions */}
                                                    {deductions.map((d, i) => {
                                                        const parseSafe = (val: string) => parseFloat(val) || 0;
                                                        const dW = unit === 'imperial' ? (parseSafe(d.l_ft || '0')) + (parseSafe(d.l_in || '0')) / 12 : parseSafe(d.l);
                                                        const dH = unit === 'imperial' ? (parseSafe(d.h_ft || '0')) + (parseSafe(d.h_in || '0')) / 12 : parseSafe(d.h);
                                                        const dCountNum = parseSafe(d.count);

                                                        return Array.from({ length: dCountNum }).map((_, idx) => {
                                                            const totalItems = deductions.reduce((acc, curr) => acc + (parseFloat(curr.count) || 0), 0);

                                                            // We need a global index to space them out
                                                            // Let's just stack them for simplicity in this ViewBox? 
                                                            // Or better: spread them out.
                                                            // Normalized Index
                                                            let globalIdx = 0;
                                                            for (let k = 0; k < i; k++) globalIdx += (parseFloat(deductions[k].count) || 0);
                                                            globalIdx += idx;

                                                            const spacing = getLength() / (totalItems + 1);
                                                            const xPos = spacing * (globalIdx + 1) - (dW / 2);
                                                            const yPos = getHeight() - dH - (getHeight() * 0.1); // 10% from bottom

                                                            return (
                                                                <g key={`${d.id}-${idx}`}>
                                                                    <rect
                                                                        x={xPos}
                                                                        y={yPos}
                                                                        width={dW}
                                                                        height={dH}
                                                                        fill={d.type === 'Door' ? '#ffffff' : '#e0f2fe'}
                                                                        stroke={d.type === 'Door' ? '#334155' : '#3b82f6'}
                                                                        strokeWidth={Math.min(getLength(), getHeight()) * 0.005}
                                                                    />
                                                                    {/* Visual Details */}
                                                                    {d.type === 'Door' ? (
                                                                        // Door Swing Arc / Line
                                                                        <path d={`M${xPos} ${yPos + dH} L${xPos + dW} ${yPos}`} stroke="#94a3b8" strokeWidth={Math.min(getLength(), getHeight()) * 0.002} strokeDasharray="0.1 0.1" fill="none" />
                                                                    ) : (
                                                                        // Window Cross
                                                                        <path d={`M${xPos + dW / 2} ${yPos} L${xPos + dW / 2} ${yPos + dH} M${xPos} ${yPos + dH / 2} L${xPos + dW} ${yPos + dH / 2}`} stroke="#bfdbfe" strokeWidth={Math.min(getLength(), getHeight()) * 0.005} />
                                                                    )}

                                                                    {/* Label */}
                                                                    <text
                                                                        x={xPos + dW / 2}
                                                                        y={yPos + dH / 2}
                                                                        textAnchor="middle"
                                                                        dominantBaseline="middle"
                                                                        fontSize={Math.min(dW, dH) * 0.3}
                                                                        fill={d.type === 'Door' ? '#64748b' : '#3b82f6'}
                                                                        fontWeight="bold"
                                                                        opacity="0.7"
                                                                    >
                                                                        {getDeductionLabel(d.type, i)}
                                                                    </text>
                                                                </g>
                                                            );
                                                        });
                                                    })}

                                                    {/* Dimensions Lines */}
                                                    {/* Width Dimension */}
                                                    <line x1="0" y1={getHeight() + getHeight() * 0.05} x2={getLength()} y2={getHeight() + getHeight() * 0.05} stroke="#64748b" strokeWidth={Math.min(getLength(), getHeight()) * 0.005} markerStart="url(#arrow)" markerEnd="url(#arrow)" />
                                                    <text x={getLength() / 2} y={getHeight() + getHeight() * 0.12} textAnchor="middle" fontSize={Math.min(getLength(), getHeight()) * 0.04} fill="#64748b" fontWeight="bold">
                                                        {unit === 'imperial' ? `${lengthFt}' ${lengthIn}"` : `${lengthM} m`}
                                                    </text>

                                                    {/* Height Dimension */}
                                                    <line x1={-getLength() * 0.05} y1="0" x2={-getLength() * 0.05} y2={getHeight()} stroke="#64748b" strokeWidth={Math.min(getLength(), getHeight()) * 0.005} markerStart="url(#arrow)" markerEnd="url(#arrow)" />
                                                    <text x={-getLength() * 0.12} y={getHeight() / 2} transform={`rotate(-90, ${-getLength() * 0.12}, ${getHeight() / 2})`} textAnchor="middle" fontSize={Math.min(getLength(), getHeight()) * 0.04} fill="#64748b" fontWeight="bold">
                                                        {unit === 'imperial' ? `${heightFt}' ${heightIn}"` : `${heightM} m`}
                                                    </text>

                                                </svg>
                                            );
                                        })()}
                                    </div>
                                )}
                            </>
                        ) : (
                            <div className="flex flex-col items-center justify-center p-8 text-slate-400 gap-2">
                                <Layers className="w-12 h-12 text-slate-300" />
                                <p className="text-xs font-mono uppercase tracking-widest">3D Volume Input</p>
                                <p className="text-[10px] opacity-70">Visual blueprint not available in volume mode.</p>
                            </div>
                        )}
                    </div>
                </div>

                {/* Bill of Quantities (BoQ) */}
                <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-sm shadow-sm">
                    <div className="bg-eng-navy px-4 py-3 flex justify-between items-center">
                        <h3 className="font-bold text-sm uppercase tracking-wide text-white flex items-center gap-2">
                            <Clipboard className="w-4 h-4" /> Bill of Quantities
                        </h3>
                        <span className="bg-white/10 px-2 py-0.5 rounded text-[10px] text-white font-mono uppercase">Est. Receipt</span>
                    </div>

                    {result ? (
                        <div className="p-0">
                            <div className="divide-y divide-slate-100 dark:divide-slate-800">
                                {/* Summary Rows */}
                                {calcMode === 'dimensions' && (
                                    <div className="p-4 grid grid-cols-2 gap-4 bg-slate-50/50 dark:bg-slate-900/50">
                                        <div>
                                            <div className="text-[10px] text-slate-400 uppercase font-bold tracking-wider">Total Area</div>
                                            <div className="font-mono font-bold text-slate-700 dark:text-slate-200">{result.totalArea} {unit === 'imperial' ? 'sq.ft' : 'm²'}</div>
                                        </div>

                                        <div>
                                            {/* NEW: Explicit Deduction Area */}
                                            <div className="text-[10px] text-red-400 uppercase font-bold tracking-wider">Deduction</div>
                                            <div className="font-mono font-bold text-red-500 dark:text-red-400">-{result.deductionArea} {unit === 'imperial' ? 'sq.ft' : 'm²'}</div>
                                        </div>

                                        <div className="col-span-2 border-t border-slate-200 dark:border-slate-700 pt-2 mt-1">
                                            <div className="flex justify-between items-center">
                                                <div className="text-[10px] text-eng-blue uppercase font-bold tracking-wider">Net Working Area</div>
                                                <div className="font-mono font-bold text-lg text-eng-blue">{result.netArea} {unit === 'imperial' ? 'sq.ft' : 'm²'}</div>
                                            </div>
                                        </div>
                                    </div>
                                )}

                                {/* Main Materials */}
                                <div className="p-4 space-y-4">
                                    <div className="flex justify-between items-center">
                                        <div className="flex items-center gap-3">
                                            <div className="w-8 h-8 rounded-full bg-red-100 dark:bg-red-900/20 text-red-600 dark:text-red-400 flex items-center justify-center">
                                                <span className="font-bold text-xs">B</span>
                                            </div>
                                            <div>
                                                <div className="text-sm font-bold text-eng-navy dark:text-white">Bricks (Nos)</div>
                                                <div className="text-[10px] text-slate-400">Class A Red Bricks</div>
                                            </div>
                                        </div>
                                        <div className="text-xl font-mono font-bold text-eng-navy dark:text-eng-blue">{result.bricks.toLocaleString()}</div>
                                    </div>

                                    <div className="flex justify-between items-center">
                                        <div className="flex items-center gap-3">
                                            <div className="w-8 h-8 rounded-full bg-gray-200 dark:bg-gray-800 text-gray-600 dark:text-gray-400 flex items-center justify-center">
                                                <span className="font-bold text-xs">C</span>
                                            </div>
                                            <div>
                                                <div className="text-sm font-bold text-eng-navy dark:text-white">Cement</div>
                                                <div className="text-[10px] text-slate-400">OPC/PPC (50kg Bags)</div>
                                            </div>
                                        </div>
                                        <div className="text-xl font-mono font-bold text-eng-navy dark:text-eng-blue">{result.cement} <span className="text-sm font-sans font-normal text-slate-400">bags</span></div>
                                    </div>

                                    <div className="flex justify-between items-center">
                                        <div className="flex items-center gap-3">
                                            <div className="w-8 h-8 rounded-full bg-yellow-100 dark:bg-yellow-900/20 text-yellow-600 dark:text-yellow-400 flex items-center justify-center">
                                                <span className="font-bold text-xs">S</span>
                                            </div>
                                            <div>
                                                <div className="text-sm font-bold text-eng-navy dark:text-white">Sand</div>
                                                <div className="text-[10px] text-slate-400">River Sand</div>
                                            </div>
                                        </div>
                                        <div className="text-xl font-mono font-bold text-eng-navy dark:text-eng-blue">{result.sand} <span className="text-sm font-sans font-normal text-slate-400">{unit === 'imperial' ? 'cu.ft' : 'm³'}</span></div>
                                    </div>

                                    {/* Water */}
                                    <div className="flex justify-between items-center">
                                        <div className="flex items-center gap-3">
                                            <div className="w-8 h-8 rounded-full bg-blue-100 dark:bg-blue-900/20 text-blue-600 dark:text-blue-400 flex items-center justify-center">
                                                <span className="font-bold text-xs">W</span>
                                            </div>
                                            <div>
                                                <div className="text-sm font-bold text-eng-navy dark:text-white">Water</div>
                                                <div className="text-[10px] text-slate-400">Clean Water (approx.)</div>
                                            </div>
                                        </div>
                                        <div className="text-xl font-mono font-bold text-eng-navy dark:text-eng-blue">{result.water} <span className="text-sm font-sans font-normal text-slate-400">Liters</span></div>
                                    </div>

                                </div>
                            </div>

                            <div className="bg-slate-50 dark:bg-slate-950 p-3 border-t border-slate-200 dark:border-slate-800 text-center">
                                <p className="text-[10px] text-slate-400 italic mb-2">Estimations are approximate.</p>
                                <button className="w-full py-2 bg-white dark:bg-slate-800 border border-slate-300 dark:border-slate-700 text-slate-600 dark:text-slate-300 text-xs font-bold uppercase tracking-wider hover:bg-slate-50 dark:hover:bg-slate-700 transition-colors flex items-center justify-center gap-2 rounded-sm shadow-sm"
                                    onClick={() => {
                                        const text = `Brick Estimate:\nBricks: ${result.bricks}\nCement: ${result.cement} Bags\nSand: ${result.sand} ${unit === 'imperial' ? 'Cu.ft' : 'm3'}\nWater: ${result.water} Liters\n(Volume: ${result.volume} m3)`;
                                        navigator.clipboard.writeText(text);
                                        alert("Estimate copied to clipboard!");
                                    }}
                                >
                                    <Clipboard className="w-3 h-3" /> Copy Estimate
                                </button>
                            </div>
                        </div>
                    ) : (
                        <div className="p-8 text-center">
                            <RefreshCw className="w-8 h-8 mx-auto text-slate-300 mb-2 animate-spin-slow" />
                            <p className="text-sm text-slate-400">Enter wall dimensions or volume to calculate.</p>
                        </div>
                    )}
                </div>

                {/* Manual Calculation Breakdown Card */}
                {result && (
                    <div className="border border-eng-blue/20 bg-slate-50/50 mt-6 rounded-sm overflow-hidden">
                        <button
                            onClick={() => setShowBreakdown(!showBreakdown)}
                            className="w-full flex items-center justify-between p-4 bg-slate-100 dark:bg-slate-800 hover:bg-slate-200 dark:hover:bg-slate-700 transition-colors"
                        >
                            <div className="flex items-center gap-2">
                                <Calculator className="w-4 h-4 text-eng-blue" />
                                <span className="font-bold text-sm uppercase tracking-wide text-eng-navy dark:text-slate-200">Manual Calculation Breakdown</span>
                            </div>
                            {showBreakdown ? <ChevronUp className="w-4 h-4 text-slate-500" /> : <ChevronDown className="w-4 h-4 text-slate-500" />}
                        </button>

                        {showBreakdown && (
                            <div className="p-5 border-t border-eng-blue/10">
                                <div className="space-y-4">
                                    {result.steps.map((step, idx) => (
                                        <div key={idx} className="border-b border-slate-200 dark:border-slate-700 last:border-0 pb-6 last:pb-0">
                                            <div className="flex justify-between items-center mb-3">
                                                <h4 className="font-bold text-sm text-eng-navy dark:text-white">{step.label}</h4>
                                                <span className="font-mono text-base font-bold text-eng-blue bg-blue-50 dark:bg-blue-900/30 px-3 py-1 rounded border border-blue-100 dark:border-blue-900/50">
                                                    {step.value}
                                                </span>
                                            </div>
                                            {step.formula && (
                                                <div className="bg-white dark:bg-slate-900 p-4 rounded-sm border border-slate-200 dark:border-slate-800 mt-2 shadow-sm">
                                                    <p className="text-[10px] text-slate-400 uppercase font-bold tracking-wider mb-2">Calculation</p>
                                                    <div className="font-mono text-sm text-slate-700 dark:text-slate-300 break-words tracking-wide leading-relaxed">
                                                        {step.formula}
                                                    </div>
                                                </div>
                                            )}
                                        </div>
                                    ))}
                                </div>
                            </div>
                        )}
                    </div>
                )}

            </div>

        </div>
    );
};
