import React, { useState, useEffect, useRef } from 'react';
import {
    Activity, Zap, Plus, Trash2,
    Bot, ChevronRight, Anchor, ArrowDownToLine,
    TrendingUp, Spline, Ruler, X, Minus, Move,
    ChevronUp
} from 'lucide-react';
import Chart from 'react-apexcharts';
import { ApexOptions } from 'apexcharts';
import { BeamConfig, AnalysisResult, ILDResult } from './beam-src/types';
import { analyzeBeam } from './beam-src/fem-engine-pro';
import { calculateInfluenceLine, ILDTarget } from './beam-src/ild-module';
import { generatePDFReport } from './beam-src/pdf-generator';
import BeamVisualizer from './beam-src/BeamVisualizer';
import ResultsCharts from './beam-src/ResultsCharts';

// ─── ILD Chart ────────────────────────────────────────────────────────────────
const ILDChart: React.FC<{ ildResult: ILDResult }> = ({ ildResult }) => {
    const series = [{ name: 'ILD Value', data: ildResult.x.map((x, i) => [x, ildResult.value[i]]) }];
    const options: ApexOptions = {
        chart: { type: 'area', toolbar: { show: false }, animations: { enabled: false }, fontFamily: 'inherit' },
        colors: ['#9333ea'], dataLabels: { enabled: false },
        stroke: { curve: 'straight', width: 2 }, fill: { type: 'solid', opacity: 0.1 },
        xaxis: {
            type: 'numeric', decimalsInFloat: 2,
            labels: { formatter: (v) => parseFloat(String(v)).toFixed(1) },
            crosshairs: { show: true, stroke: { color: '#94a3b8', dashArray: 4 } },
            tooltip: { enabled: false }
        },
        yaxis: { decimalsInFloat: 3, labels: { style: { colors: '#64748b' } } },
        grid: { borderColor: '#f1f5f9', strokeDashArray: 3 },
        tooltip: {
            theme: 'light',
            x: { formatter: (v: number) => `x = ${v.toFixed(2)}m` },
            y: { formatter: (v: number) => `${v.toFixed(3)}` }
        }
    };
    return <Chart options={options} series={series} type="area" height="100%" width="100%" />;
};

// ─── Initial Config ────────────────────────────────────────────────────────────
const getInitialBeamConfig = (): BeamConfig => ({
    length: 10, E: 200000000, I: 0.0004, depth: 0.5,
    supports: [{ id: 's1', type: 'pin', x: 0 }, { id: 's2', type: 'roller', x: 10 }],
    loads: [], hinges: [], sections: []
});

// ─── Math evaluator ───────────────────────────────────────────────────────────
const safeEvaluateMath = (expr: string): number | null => {
    try {
        const sanitized = expr.toLowerCase().replace(/[^0-9+\-*/().e pi sin cos tan sqrt abs \^]/g, '');
        if (!sanitized) return null;
        const finalExpr = sanitized
            .replace(/pi/g, 'Math.PI').replace(/sin/g, 'Math.sin').replace(/cos/g, 'Math.cos')
            .replace(/tan/g, 'Math.tan').replace(/sqrt/g, 'Math.sqrt').replace(/abs/g, 'Math.abs')
            .replace(/\^/g, '**');
        // eslint-disable-next-line no-new-func
        const result = new Function(`return ${finalExpr}`)();
        return typeof result === 'number' && !isNaN(result) && isFinite(result)
            ? parseFloat(result.toPrecision(12)) : null;
    } catch { return null; }
};

const getNextId = (list: any[]) => {
    const nums = list.map(i => { const m = i.id.match(/(\d+)$/); return m ? parseInt(m[0]) : NaN; }).filter(n => !isNaN(n));
    return nums.length > 0 ? (Math.max(...nums) + 1).toString() : '1';
};

// ─── UI Primitives ────────────────────────────────────────────────────────────
const Label: React.FC<{ children?: React.ReactNode }> = ({ children }) => (
    <label className="block text-[11px] font-semibold text-slate-400 uppercase tracking-widest mb-1.5">{children}</label>
);

const DarkInput = ({ value, onChange, suffix, className = '', ...props }: any) => {
    const [local, setLocal] = useState(value?.toString() ?? '');
    const [focused, setFocused] = useState(false);
    useEffect(() => { if (!focused) setLocal(value?.toString() ?? ''); }, [value, focused]);
    const eval_ = () => {
        const r = safeEvaluateMath(local);
        if (r !== null) { setLocal(r.toString()); onChange(r); }
        else if (local === '' || local === '-') onChange(0);
        else setLocal(value?.toString() ?? '');
    };
    return (
        <div className="relative flex items-center">
            <input
                type="text" inputMode="decimal" value={local}
                onFocus={() => setFocused(true)}
                onBlur={() => { setFocused(false); eval_(); }}
                onKeyDown={e => { if (e.key === 'Enter') e.currentTarget.blur(); }}
                onChange={e => setLocal(e.target.value)}
                className={`w-full bg-slate-800 border border-slate-700 text-slate-100 text-sm rounded-lg pl-3 py-3 outline-none font-mono transition-all focus:ring-2 focus:ring-blue-500 focus:border-blue-400 ${suffix ? 'pr-16' : 'pr-3'} ${className}`}
                autoComplete="off" {...props}
            />
            {suffix && (
                <div className="absolute right-1 top-1 bottom-1 px-2 flex items-center bg-slate-700 rounded-md text-[11px] text-slate-400 font-medium pointer-events-none">
                    {suffix}
                </div>
            )}
        </div>
    );
};

const DarkSelect = ({ children, ...props }: any) => (
    <div className="relative">
        <select
            className="appearance-none w-full bg-slate-800 border border-slate-700 text-slate-100 text-sm rounded-lg px-3 py-3 pr-9 focus:ring-2 focus:ring-blue-500 outline-none font-medium"
            {...props}
        >{children}</select>
        <div className="pointer-events-none absolute inset-y-0 right-0 flex items-center px-3 text-slate-400">
            <ChevronRight size={13} className="rotate-90" />
        </div>
    </div>
);

const PrimaryButton = ({ onClick, children, icon: Icon, loading }: any) => (
    <button onClick={onClick} disabled={loading}
        className="w-full py-3.5 bg-blue-600 hover:bg-blue-500 active:bg-blue-700 text-white text-sm font-bold rounded-xl flex items-center justify-center gap-2 transition-all shadow-lg shadow-blue-900/30 disabled:opacity-60 touch-manipulation">
        {Icon && <Icon size={16} />}{children}
    </button>
);

const SectionCard: React.FC<{ children: React.ReactNode; className?: string }> = ({ children, className = '' }) => (
    <div className={`bg-slate-800/60 border border-slate-700/60 rounded-xl p-4 ${className}`}>{children}</div>
);

// ─── Tool nav items ───────────────────────────────────────────────────────────
type ToolId = 'properties' | 'hinges' | 'supports' | 'loads' | 'ild' | 'results';
const TOOL_ITEMS: { id: ToolId; icon: React.ElementType; label: string; color: string }[] = [
    { id: 'properties', icon: Ruler, label: 'Properties', color: 'blue' },
    { id: 'hinges', icon: Spline, label: 'Hinges', color: 'blue' },
    { id: 'supports', icon: Anchor, label: 'Supports', color: 'blue' },
    { id: 'loads', icon: ArrowDownToLine, label: 'Loads', color: 'blue' },
    { id: 'ild', icon: TrendingUp, label: 'ILD', color: 'purple' },
    { id: 'results', icon: Zap, label: 'Results', color: 'emerald' },
];

const activeClass = (id: ToolId, active: ToolId) => {
    const item = TOOL_ITEMS.find(t => t.id === id);
    if (id !== active) return 'text-slate-500 hover:text-slate-200 hover:bg-slate-800';
    return item?.color === 'purple'
        ? 'bg-purple-600 text-white shadow-lg shadow-purple-900/40'
        : item?.color === 'emerald'
            ? 'bg-emerald-600 text-white shadow-lg shadow-emerald-900/40'
            : 'bg-blue-600 text-white shadow-lg shadow-blue-900/40';
};

// ─── Main Component ───────────────────────────────────────────────────────────
export const BeamCalculator: React.FC = () => {
    const [activeTool, setActiveTool] = useState<ToolId>('properties');
    const [selectedId, setSelectedId] = useState<string | null>(null);
    // Mobile: panel is a bottom-sheet; desktop: always-visible sidebar
    const [panelOpen, setPanelOpen] = useState(false);

    const [beamConfig, setBeamConfig] = useState<BeamConfig>(getInitialBeamConfig());
    const [beamResults, setBeamResults] = useState<AnalysisResult | null>(null);
    const [ildTarget, setIldTarget] = useState<ILDTarget>({ type: 'shear', location: 2, component: 'Fy' });
    const [ildResult, setIldResult] = useState<ILDResult | null>(null);
    const [viewConfig, setViewConfig] = useState({ zoom: 1, pan: { x: 0, y: 0 } });
    const [isPdfLoading, setIsPdfLoading] = useState(false);

    useEffect(() => { setSelectedId(null); }, [activeTool]);
    useEffect(() => {
        try {
            setBeamResults(analyzeBeam(beamConfig));
            if (activeTool === 'ild') setIldResult(calculateInfluenceLine(beamConfig, ildTarget));
        } catch { setBeamResults(null); }
    }, [beamConfig, ildTarget, activeTool]);

    const handleToolSelect = (id: ToolId) => {
        setActiveTool(id);
        setPanelOpen(true);
    };

    const handleDelete = (id: string, list: 'loads' | 'supports' | 'hinges' | 'sections') => {
        setBeamConfig(p => ({ ...p, [list]: (p as any)[list].filter((x: any) => x.id !== id) }));
        setSelectedId(null);
    };

    const zoomIn = () => setViewConfig(p => ({ ...p, zoom: Math.min(5, p.zoom * 1.2) }));
    const zoomOut = () => setViewConfig(p => ({ ...p, zoom: Math.max(0.2, p.zoom / 1.2) }));
    const fit = () => setViewConfig({ zoom: 1, pan: { x: 0, y: 0 } });

    // ── Panel content ─────────────────────────────────────────────────────────
    const renderProperties = () => {
        const sections = beamConfig.sections || [];
        return (
            <div className="space-y-5">
                <SectionCard>
                    <Label>Beam Length</Label>
                    <DarkInput value={beamConfig.length} onChange={(v: number) => setBeamConfig(p => ({ ...p, length: Math.max(0.1, v) }))} suffix="m" />
                    <p className="text-[10px] text-slate-500 mt-1.5">Total length of the continuous beam.</p>
                </SectionCard>
                <SectionCard>
                    <Label>Section Properties</Label>
                    <div className="grid grid-cols-2 gap-3">
                        <div>
                            <p className="text-[10px] text-slate-500 mb-1">Modulus (E)</p>
                            <DarkInput value={beamConfig.E} onChange={(v: number) => setBeamConfig(p => ({ ...p, E: v }))} suffix="kPa" />
                        </div>
                        <div>
                            <p className="text-[10px] text-slate-500 mb-1">Inertia (I)</p>
                            <DarkInput value={beamConfig.I} onChange={(v: number) => setBeamConfig(p => ({ ...p, I: v }))} suffix="m⁴" />
                        </div>
                        <div className="col-span-2">
                            <p className="text-[10px] text-slate-500 mb-1">Depth (d)</p>
                            <DarkInput value={beamConfig.depth} onChange={(v: number) => setBeamConfig(p => ({ ...p, depth: v }))} suffix="m" />
                        </div>
                    </div>
                </SectionCard>
                <div>
                    <div className="flex justify-between items-center mb-2">
                        <Label>Variable Sections</Label>
                        <button onClick={() => setBeamConfig(p => ({ ...p, sections: [...(p.sections || []), { x: 0, end: p.length / 2, E: p.E, I: p.I * 2, depth: (p.depth || 0.5) * 1.2 }] }))}
                            className="p-2 bg-slate-700 text-blue-400 rounded-lg hover:bg-slate-600 transition-colors touch-manipulation">
                            <Plus size={14} />
                        </button>
                    </div>
                    {sections.length === 0
                        ? <p className="text-xs text-slate-500 italic text-center py-3">Beam is prismatic (uniform section).</p>
                        : <div className="space-y-3">
                            {sections.map((sec, i) => (
                                <SectionCard key={i}>
                                    <div className="flex justify-between items-center mb-3">
                                        <span className="text-xs font-bold text-slate-300">Segment {i + 1}</span>
                                        <button onClick={() => setBeamConfig(p => ({ ...p, sections: p.sections?.filter((_, idx) => idx !== i) }))}
                                            className="p-1.5 text-red-400 hover:bg-red-900/30 rounded-lg touch-manipulation"><Trash2 size={13} /></button>
                                    </div>
                                    <div className="grid grid-cols-2 gap-2">
                                        <DarkInput value={sec.x} onChange={(v: number) => setBeamConfig(p => ({ ...p, sections: p.sections?.map((s, idx) => idx === i ? { ...s, x: v } : s) }))} suffix="start" />
                                        <DarkInput value={sec.end} onChange={(v: number) => setBeamConfig(p => ({ ...p, sections: p.sections?.map((s, idx) => idx === i ? { ...s, end: v } : s) }))} suffix="end" />
                                        <DarkInput value={sec.E} onChange={(v: number) => setBeamConfig(p => ({ ...p, sections: p.sections?.map((s, idx) => idx === i ? { ...s, E: v } : s) }))} suffix="E" />
                                        <DarkInput value={sec.I} onChange={(v: number) => setBeamConfig(p => ({ ...p, sections: p.sections?.map((s, idx) => idx === i ? { ...s, I: v } : s) }))} suffix="I" />
                                    </div>
                                </SectionCard>
                            ))}
                        </div>
                    }
                </div>
            </div>
        );
    };

    const renderHinges = () => {
        const hinges = beamConfig.hinges || [];
        const sel = hinges.find(h => h.id === selectedId);
        return (
            <div className="space-y-4">
                {sel ? (
                    <SectionCard>
                        <div className="flex justify-between items-center mb-4">
                            <span className="text-white font-mono font-bold">{sel.id}</span>
                            <button onClick={() => handleDelete(sel.id, 'hinges')} className="p-2 bg-red-900/30 text-red-400 rounded-lg touch-manipulation"><Trash2 size={16} /></button>
                        </div>
                        <Label>Location (x)</Label>
                        <DarkInput value={sel.x} onChange={(v: number) => setBeamConfig(p => ({ ...p, hinges: p.hinges.map(h => h.id === sel.id ? { ...h, x: v } : h) }))} suffix="m" />
                        <button onClick={() => setSelectedId(null)} className="w-full mt-4 py-3 text-xs font-bold text-slate-300 border border-slate-600 rounded-xl hover:bg-slate-700 transition-colors touch-manipulation">Done Editing</button>
                    </SectionCard>
                ) : (
                    <PrimaryButton icon={Plus} onClick={() => {
                        const newId = `h${getNextId(hinges)}`;
                        setBeamConfig(p => ({ ...p, hinges: [...p.hinges, { id: newId, x: p.length / 2 }] }));
                        setSelectedId(newId);
                    }}>Add Internal Hinge</PrimaryButton>
                )}
                {!sel && hinges.length > 0 && (
                    <div className="space-y-2">
                        <Label>Existing Hinges</Label>
                        {hinges.map(h => (
                            <button key={h.id} onClick={() => setSelectedId(h.id)}
                                className="w-full flex items-center justify-between px-4 py-3 bg-slate-800 hover:bg-slate-700 rounded-xl border border-slate-700 transition-colors touch-manipulation">
                                <span className="text-sm text-slate-300 font-mono">{h.id}</span>
                                <span className="text-sm text-blue-400 font-bold">x = {h.x}m</span>
                            </button>
                        ))}
                    </div>
                )}
            </div>
        );
    };

    const renderSupports = () => {
        const supports = beamConfig.supports;
        const sel = supports.find(s => s.id === selectedId);
        if (sel) return (
            <div className="space-y-5">
                <div className="flex items-center justify-between">
                    <div>
                        <p className="text-[11px] text-slate-400 uppercase font-semibold tracking-widest">Editing</p>
                        <p className="text-2xl font-bold text-white font-mono">{sel.id}</p>
                    </div>
                    <button onClick={() => handleDelete(sel.id, 'supports')} className="p-3 bg-red-900/20 text-red-400 rounded-xl touch-manipulation"><Trash2 size={18} /></button>
                </div>
                <SectionCard>
                    <Label>Support Type</Label>
                    <div className="grid grid-cols-2 gap-2">
                        {['pin', 'roller', 'fixed', 'spring_y'].map(t => (
                            <button key={t} onClick={() => setBeamConfig(p => ({ ...p, supports: p.supports.map(s => s.id === sel.id ? { ...s, type: t as any } : s) }))}
                                className={`py-3 text-sm font-bold rounded-xl border touch-manipulation ${sel.type === t ? 'bg-blue-600 border-blue-500 text-white' : 'bg-slate-800 border-slate-700 text-slate-400 hover:bg-slate-700'}`}>
                                {t}
                            </button>
                        ))}
                    </div>
                </SectionCard>
                <SectionCard>
                    <Label>Position (x)</Label>
                    <DarkInput value={sel.x} onChange={(v: number) => setBeamConfig(p => ({ ...p, supports: p.supports.map(s => s.id === sel.id ? { ...s, x: v } : s) }))} suffix="m" />
                </SectionCard>
                {sel.type === 'spring_y' && (
                    <SectionCard>
                        <Label>Stiffness (k)</Label>
                        <DarkInput value={sel.stiffness ?? 0} onChange={(v: number) => setBeamConfig(p => ({ ...p, supports: p.supports.map(s => s.id === sel.id ? { ...s, stiffness: v } : s) }))} suffix="kN/m" />
                    </SectionCard>
                )}
                <SectionCard>
                    <Label>Settlement (dy)</Label>
                    <DarkInput value={sel.settlement ?? 0} onChange={(v: number) => setBeamConfig(p => ({ ...p, supports: p.supports.map(s => s.id === sel.id ? { ...s, settlement: v } : s) }))} suffix="m" />
                </SectionCard>
                <button onClick={() => setSelectedId(null)} className="w-full py-3 text-sm font-bold text-slate-300 border border-slate-600 rounded-xl hover:bg-slate-700 transition-colors touch-manipulation">Done</button>
            </div>
        );
        return (
            <div className="space-y-4">
                <div>
                    <Label>Add Support</Label>
                    <div className="grid grid-cols-2 gap-3 mt-2">
                        {['pin', 'roller', 'fixed', 'spring_y'].map(type => (
                            <button key={type} onClick={() => {
                                const newId = `s${getNextId(supports)}`;
                                setBeamConfig(p => ({ ...p, supports: [...p.supports, { id: newId, type: type as any, x: p.length / 2 }] }));
                                setSelectedId(newId);
                            }} className="py-4 bg-slate-800 border border-slate-700 rounded-xl hover:bg-slate-700 hover:border-slate-500 transition-all touch-manipulation">
                                <span className="text-sm font-bold text-slate-300 uppercase">{type}</span>
                            </button>
                        ))}
                    </div>
                </div>
                {supports.length > 0 && (
                    <div className="space-y-2">
                        <Label>Existing Supports</Label>
                        {supports.map(s => (
                            <button key={s.id} onClick={() => setSelectedId(s.id)}
                                className="w-full flex items-center justify-between px-4 py-3 bg-slate-800 hover:bg-slate-700 rounded-xl border border-slate-700 transition-colors touch-manipulation">
                                <span className="text-sm text-slate-300 font-mono">{s.id} <span className="text-slate-500">({s.type})</span></span>
                                <span className="text-sm text-blue-400 font-bold">x = {s.x}m</span>
                            </button>
                        ))}
                    </div>
                )}
            </div>
        );
    };

    const renderLoads = () => {
        const loads = beamConfig.loads;
        const sel = loads.find(l => l.id === selectedId);
        if (sel) return (
            <div className="space-y-5">
                <div className="flex items-center justify-between">
                    <div>
                        <p className="text-[11px] text-slate-400 uppercase font-semibold tracking-widest">Editing Load</p>
                        <p className="text-2xl font-bold text-white font-mono">{sel.id}</p>
                        <p className="text-xs text-blue-400 font-bold uppercase mt-0.5">{sel.type}</p>
                    </div>
                    <button onClick={() => handleDelete(sel.id, 'loads')} className="p-3 bg-red-900/20 text-red-400 rounded-xl touch-manipulation"><Trash2 size={18} /></button>
                </div>
                {(sel.type === 'point' || sel.type === 'moment') && (
                    <SectionCard>
                        <Label>Position (x)</Label>
                        <DarkInput value={sel.x ?? 0} onChange={(v: number) => setBeamConfig(p => ({ ...p, loads: p.loads.map(l => l.id === sel.id ? { ...l, x: v } : l) }))} suffix="m" />
                    </SectionCard>
                )}
                {(sel.type === 'udl' || sel.type === 'uvl') && (
                    <SectionCard>
                        <div className="grid grid-cols-2 gap-3">
                            <div><Label>Start</Label><DarkInput value={sel.start ?? 0} onChange={(v: number) => setBeamConfig(p => ({ ...p, loads: p.loads.map(l => l.id === sel.id ? { ...l, start: v } : l) }))} suffix="m" /></div>
                            <div><Label>End</Label><DarkInput value={sel.end ?? 0} onChange={(v: number) => setBeamConfig(p => ({ ...p, loads: p.loads.map(l => l.id === sel.id ? { ...l, end: v } : l) }))} suffix="m" /></div>
                        </div>
                    </SectionCard>
                )}
                <SectionCard>
                    <Label>Magnitude</Label>
                    {sel.type === 'uvl' ? (
                        <div className="grid grid-cols-2 gap-3">
                            <DarkInput value={sel.valueStart ?? 0} onChange={(v: number) => setBeamConfig(p => ({ ...p, loads: p.loads.map(l => l.id === sel.id ? { ...l, valueStart: v } : l) }))} suffix="w₁" />
                            <DarkInput value={sel.valueEnd ?? 0} onChange={(v: number) => setBeamConfig(p => ({ ...p, loads: p.loads.map(l => l.id === sel.id ? { ...l, valueEnd: v } : l) }))} suffix="w₂" />
                        </div>
                    ) : (
                        <DarkInput value={sel.value} onChange={(v: number) => setBeamConfig(p => ({ ...p, loads: p.loads.map(l => l.id === sel.id ? { ...l, value: v } : l) }))} suffix={sel.type === 'moment' ? 'kNm' : 'kN'} />
                    )}
                </SectionCard>
                <button onClick={() => setSelectedId(null)} className="w-full py-3 text-sm font-bold text-slate-300 border border-slate-600 rounded-xl hover:bg-slate-700 transition-colors touch-manipulation">Done</button>
            </div>
        );
        return (
            <div className="space-y-4">
                <div>
                    <Label>Add Load</Label>
                    <div className="grid grid-cols-2 gap-3 mt-2">
                        {['point', 'moment', 'udl', 'uvl'].map(type => (
                            <button key={type} onClick={() => {
                                const newId = `l${getNextId(loads)}`;
                                let nl: any = { id: newId, type, value: -10 };
                                if (type === 'point' || type === 'moment') nl.x = beamConfig.length / 2;
                                if (type === 'udl') { nl.start = 0; nl.end = beamConfig.length; nl.value = -5; }
                                if (type === 'uvl') { nl.start = 0; nl.end = beamConfig.length; nl.valueStart = -5; nl.valueEnd = -10; }
                                setBeamConfig(p => ({ ...p, loads: [...p.loads, nl] }));
                                setSelectedId(newId);
                            }} className="py-4 bg-slate-800 border border-slate-700 rounded-xl hover:bg-slate-700 hover:border-slate-500 transition-all touch-manipulation">
                                <span className="text-sm font-bold text-slate-300 uppercase">{type}</span>
                            </button>
                        ))}
                    </div>
                </div>
                {loads.length > 0 && (
                    <div className="space-y-2">
                        <Label>Existing Loads</Label>
                        {loads.map(l => (
                            <button key={l.id} onClick={() => setSelectedId(l.id)}
                                className="w-full flex items-center justify-between px-4 py-3 bg-slate-800 hover:bg-slate-700 rounded-xl border border-slate-700 transition-colors touch-manipulation">
                                <span className="text-sm text-slate-300 font-mono">{l.id} <span className="text-slate-500">({l.type})</span></span>
                                <span className="text-sm text-blue-400 font-bold">{l.type.includes('u') ? `${l.start}–${l.end}m` : `@${l.x}m`}</span>
                            </button>
                        ))}
                    </div>
                )}
            </div>
        );
    };

    const renderILD = () => (
        <div className="space-y-4">
            <SectionCard>
                <Label>Target Response</Label>
                <DarkSelect value={ildTarget.type} onChange={(e: any) => setIldTarget(p => ({ ...p, type: e.target.value }))}>
                    <option value="reaction">Reaction</option>
                    <option value="shear">Shear Force</option>
                    <option value="moment">Bending Moment</option>
                    <option value="deflection">Deflection</option>
                </DarkSelect>
            </SectionCard>
            <SectionCard>
                <Label>Location (x)</Label>
                <DarkInput value={ildTarget.location} onChange={(v: number) => setIldTarget(p => ({ ...p, location: v }))} suffix="m" />
            </SectionCard>
            {ildTarget.type === 'reaction' && (
                <SectionCard>
                    <Label>Component</Label>
                    <DarkSelect value={ildTarget.component} onChange={(e: any) => setIldTarget(p => ({ ...p, component: e.target.value }))}>
                        <option value="Fy">Vertical (Fy)</option>
                        <option value="Mz">Moment (Mz)</option>
                    </DarkSelect>
                </SectionCard>
            )}
            <div className="text-xs text-slate-500 text-center py-1">Tap the beam diagram to pick a location</div>
        </div>
    );

    const renderResultsPanel = () => (
        <div className="space-y-4">
            {beamResults ? (
                <>
                    {/* Quick summary cards */}
                    <div className="grid grid-cols-2 gap-3">
                        {[
                            { label: 'Max Shear', val: beamResults.summary.maxShear.value.toFixed(2), unit: 'kN', color: 'text-emerald-400' },
                            { label: 'Max Moment', val: beamResults.summary.maxMoment.value.toFixed(2), unit: 'kNm', color: 'text-blue-400' },
                            { label: 'Max Deflection', val: (beamResults.summary.maxDeflection.value * 1000).toFixed(3), unit: 'mm', color: 'text-amber-400' },
                            { label: 'Max Stress', val: (beamResults.summary.maxStress.value / 1000).toFixed(2), unit: 'MPa', color: 'text-purple-400' },
                        ].map(({ label, val, unit, color }) => (
                            <SectionCard key={label}>
                                <p className="text-[10px] text-slate-500 uppercase font-bold tracking-widest mb-1">{label}</p>
                                <p className={`text-lg font-bold font-mono ${color}`}>{val}</p>
                                <p className="text-[10px] text-slate-600 font-medium">{unit}</p>
                            </SectionCard>
                        ))}
                    </div>
                    <PrimaryButton icon={Bot} loading={isPdfLoading} onClick={async () => {
                        setIsPdfLoading(true);
                        await generatePDFReport(beamConfig, beamResults, 'beam-printable-visualizer', 'beam-printable-charts');
                        setIsPdfLoading(false);
                    }}>
                        {isPdfLoading ? 'Generating PDF…' : 'Download PDF Report'}
                    </PrimaryButton>
                </>
            ) : (
                <div className="text-center py-8 text-slate-500">
                    <Activity size={32} className="mx-auto opacity-30 mb-2" />
                    <p className="text-sm">Configure your structure first.</p>
                </div>
            )}
        </div>
    );

    const panelContent = () => {
        const item = TOOL_ITEMS.find(t => t.id === activeTool)!;
        return (
            <div className="flex flex-col h-full">
                {/* Panel Header */}
                <div className="flex items-center justify-between px-5 py-4 border-b border-slate-800 shrink-0">
                    <div className="flex items-center gap-3">
                        <div className={`p-2 rounded-lg ${activeTool === 'ild' ? 'bg-purple-900/50 text-purple-400' : activeTool === 'results' ? 'bg-emerald-900/50 text-emerald-400' : 'bg-blue-900/50 text-blue-400'}`}>
                            <item.icon size={16} />
                        </div>
                        <span className="font-bold text-slate-100 text-sm">{item.label}</span>
                    </div>
                    <button onClick={() => setPanelOpen(false)} className="md:hidden p-2 text-slate-400 hover:text-white hover:bg-slate-800 rounded-xl touch-manipulation">
                        <X size={18} />
                    </button>
                </div>
                {/* Scrollable Content */}
                <div className="flex-1 overflow-y-auto p-5 pb-8">
                    {activeTool === 'properties' && renderProperties()}
                    {activeTool === 'hinges' && renderHinges()}
                    {activeTool === 'supports' && renderSupports()}
                    {activeTool === 'loads' && renderLoads()}
                    {activeTool === 'ild' && renderILD()}
                    {activeTool === 'results' && renderResultsPanel()}
                </div>
            </div>
        );
    };

    return (
        <div className="flex flex-col bg-slate-900 font-sans text-slate-100 rounded-2xl border border-slate-700 shadow-2xl overflow-hidden"
            style={{ height: 'clamp(600px, 85vh, 900px)' }}>

            {/* ── TOP BAR (mobile only) ─────────────────────────────────────── */}
            <div className="md:hidden flex items-center justify-between px-4 py-3 bg-slate-950 border-b border-slate-800 shrink-0">
                <div className="flex items-center gap-2">
                    <div className="w-7 h-7 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-lg flex items-center justify-center">
                        <Activity size={14} className="text-white" />
                    </div>
                    <span className="font-bold text-sm text-white">Beam Calculator</span>
                </div>
                <button onClick={() => setPanelOpen(true)}
                    className="flex items-center gap-1.5 px-3 py-1.5 bg-blue-600 text-white text-xs font-bold rounded-lg touch-manipulation">
                    <ChevronUp size={13} /> Edit
                </button>
            </div>

            <div className="flex flex-1 min-h-0">
                {/* ── LEFT ICON RAIL (desktop only) ─────────────────────────────── */}
                <div className="hidden md:flex flex-col items-center gap-1 w-16 bg-slate-950 border-r border-slate-800 py-4 shrink-0">
                    <div className="w-9 h-9 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-xl flex items-center justify-center mb-5 shadow-lg">
                        <Activity size={18} className="text-white" />
                    </div>
                    {TOOL_ITEMS.map(({ id, icon: Icon }) => (
                        <button key={id} onClick={() => handleToolSelect(id)}
                            className={`p-3 rounded-xl transition-all ${activeClass(id, activeTool)}`}
                            title={id}>
                            <Icon size={20} />
                        </button>
                    ))}
                </div>

                {/* ── INSPECTOR PANEL (desktop: always visible; mobile: slide-up sheet) ── */}
                {/* Desktop sidebar */}
                <div className="hidden md:flex flex-col w-72 bg-slate-900 border-r border-slate-800 shrink-0">
                    {panelContent()}
                </div>

                {/* Mobile: backdrop + bottom sheet */}
                {panelOpen && (
                    <div className="md:hidden fixed inset-0 z-50 flex flex-col justify-end">
                        {/* Backdrop */}
                        <div className="absolute inset-0 bg-black/60" onClick={() => setPanelOpen(false)} />
                        {/* Sheet */}
                        <div className="relative bg-slate-900 rounded-t-3xl border-t border-slate-700 flex flex-col z-10"
                            style={{ maxHeight: '88vh' }}>
                            {/* Drag handle */}
                            <div className="flex justify-center pt-3 pb-1 shrink-0">
                                <div className="w-10 h-1 bg-slate-600 rounded-full" />
                            </div>
                            <div className="flex-1 overflow-hidden flex flex-col">
                                {panelContent()}
                            </div>
                        </div>
                    </div>
                )}

                {/* ── MAIN CANVAS ───────────────────────────────────────────────── */}
                <div className="flex-1 flex flex-col bg-slate-100 min-w-0 overflow-hidden">
                    {activeTool === 'results' ? (
                        /* Results view: visualizer on top, charts below */
                        <div className="flex-1 flex flex-col overflow-y-auto">
                            {/* Beam diagram */}
                            <div className="relative shrink-0 bg-white border-b border-slate-200 shadow-sm" style={{ height: 'clamp(160px, 35vh, 300px)' }}>
                                <div id="beam-printable-visualizer" className="absolute inset-2 md:inset-4 bg-white rounded-xl shadow border border-slate-100 overflow-hidden">
                                    <div className="absolute top-2 right-2 z-10">
                                        <div className="bg-white shadow border border-slate-200 rounded-xl flex flex-col p-1 gap-0.5">
                                            <button onClick={zoomIn} className="p-2 hover:bg-slate-100 rounded-lg text-slate-600 touch-manipulation"><Plus size={14} /></button>
                                            <button onClick={zoomOut} className="p-2 hover:bg-slate-100 rounded-lg text-slate-600 touch-manipulation"><Minus size={14} /></button>
                                            <button onClick={fit} className="p-2 hover:bg-slate-100 rounded-lg text-slate-600 touch-manipulation"><Move size={14} /></button>
                                        </div>
                                    </div>
                                    <div className="w-full h-full p-3 md:p-6 flex items-center justify-center">
                                        <div className="w-full" style={{ aspectRatio: '2/1', minHeight: 80 }}>
                                            <BeamVisualizer config={beamConfig} selectedILDPoint={null} zoom={viewConfig.zoom} pan={viewConfig.pan} />
                                        </div>
                                    </div>
                                </div>
                            </div>
                            {/* Charts */}
                            <div className="flex-1 p-3 md:p-6">
                                <h2 className="text-sm font-bold text-slate-800 mb-4 flex items-center gap-2 uppercase tracking-wide">
                                    <Activity size={14} className="text-blue-600" /> Analysis Results
                                </h2>
                                {beamResults ? (
                                    <div id="beam-printable-charts" className="space-y-3 md:space-y-5">
                                        <ResultsCharts results={beamResults} config={beamConfig} />
                                    </div>
                                ) : (
                                    <div className="flex flex-col items-center justify-center h-32 text-slate-400 bg-white rounded-xl border border-dashed border-slate-200">
                                        <Activity size={18} className="opacity-40 mb-2" />
                                        <p className="text-sm">No results yet — configure your beam.</p>
                                    </div>
                                )}
                            </div>
                        </div>
                    ) : (
                        /* Normal editing view */
                        <div className="flex-1 flex flex-col min-h-0">
                            {/* Beam visualizer */}
                            <div className="flex-1 relative bg-slate-100 min-h-0">
                                {/* Zoom controls */}
                                <div className="absolute top-2 right-2 z-10">
                                    <div className="bg-white shadow-md border border-slate-200 rounded-xl flex flex-col p-1 gap-0.5">
                                        <button onClick={zoomIn} className="p-2.5 hover:bg-slate-100 rounded-xl text-slate-600 touch-manipulation"><Plus size={16} /></button>
                                        <button onClick={zoomOut} className="p-2.5 hover:bg-slate-100 rounded-xl text-slate-600 touch-manipulation"><Minus size={16} /></button>
                                        <button onClick={fit} className="p-2.5 hover:bg-slate-100 rounded-xl text-slate-600 touch-manipulation"><Move size={16} /></button>
                                    </div>
                                </div>
                                <div className="w-full h-full p-3 md:p-8 flex items-center justify-center">
                                    <div className="w-full max-w-4xl" style={{ aspectRatio: '2/1', minHeight: 140 }}>
                                        <BeamVisualizer
                                            config={beamConfig}
                                            selectedILDPoint={activeTool === 'ild' && ildTarget.type !== 'reaction' ? ildTarget.location : null}
                                            onSelectILDPoint={activeTool === 'ild' && ildTarget.type !== 'reaction' ? x => setIldTarget(p => ({ ...p, location: x })) : undefined}
                                            zoom={viewConfig.zoom} pan={viewConfig.pan}
                                        />
                                    </div>
                                </div>
                            </div>
                            {/* ILD chart (when active) */}
                            {activeTool === 'ild' && ildResult && (
                                <div className="shrink-0 border-t border-slate-200 bg-white relative shadow-[0_-4px_12px_rgba(0,0,0,0.06)]" style={{ height: 'clamp(160px, 30vh, 260px)' }}>
                                    <div className="absolute top-0 left-0 w-full h-0.5 bg-gradient-to-r from-blue-500 via-indigo-500 to-purple-500" />
                                    <div className="p-3 h-full flex flex-col">
                                        <h3 className="text-[11px] font-bold text-slate-700 mb-1.5 uppercase tracking-wider flex items-center gap-1.5">
                                            <TrendingUp size={12} className="text-purple-600" />
                                            ILD — {ildTarget.type} @ {ildTarget.location}m
                                        </h3>
                                        <div className="flex-1 min-h-0"><ILDChart ildResult={ildResult} /></div>
                                    </div>
                                </div>
                            )}
                        </div>
                    )}
                </div>
            </div>

            {/* ── BOTTOM ICON NAV (mobile only) ─────────────────────────────── */}
            <div className="md:hidden flex items-center bg-slate-950 border-t border-slate-800 shrink-0">
                {TOOL_ITEMS.map(({ id, icon: Icon, label }) => {
                    const isActive = activeTool === id;
                    return (
                        <button key={id} onClick={() => handleToolSelect(id)}
                            className={`flex-1 flex flex-col items-center gap-0.5 py-2.5 transition-all touch-manipulation ${isActive ? 'text-white' : 'text-slate-500'}`}>
                            <div className={`p-1.5 rounded-lg ${isActive ? activeClass(id, activeTool) : ''}`}>
                                <Icon size={18} />
                            </div>
                            <span className={`text-[9px] font-bold uppercase tracking-wide ${isActive ? 'text-blue-400' : 'text-slate-600'}`}>{label}</span>
                        </button>
                    );
                })}
            </div>
        </div>
    );
};
