import React, { useState } from 'react';
import { BrickCalculator } from './tools/BrickCalculator';
import { ConcreteCalculator } from './tools/ConcreteCalculator';
import { PlasterEstimator } from './tools/PlasterEstimator';
import { StoneMasonryCalculator } from './tools/StoneMasonryCalculator';
import { BeamCalculator } from './tools/BeamCalculator';
import { Wrench, ChevronRight } from 'lucide-react';

const TOOLS = [
    { id: 'brick', name: 'Brick Masonry Calculator', desc: 'Estimate bricks, cement, and sand for walls.', Component: BrickCalculator },
    { id: 'concrete', name: 'Concrete Calculator', desc: 'Estimate Beam, Column, and Block materials.', Component: ConcreteCalculator },
    { id: 'plaster', name: 'Plaster Estimator', desc: 'Estimate cement and sand for wall and ceiling plastering.', Component: PlasterEstimator },
    { id: 'stone', name: 'Stone Masonry Calculator', desc: 'Estimate stone, cement, and sand for foundations and walls.', Component: StoneMasonryCalculator },
    { id: 'beam', name: 'Beam Calculator', desc: 'Full FEM-based structural beam analysis with shear, moment, deflection, and influence line diagrams.', Component: BeamCalculator },
];

export const ToolsPage: React.FC = () => {
    const [activeToolId, setActiveToolId] = useState<string>(TOOLS[0].id);

    const ActiveComponent = TOOLS.find(t => t.id === activeToolId)?.Component || BrickCalculator;

    return (
        <section className="py-24 min-h-screen bg-white dark:bg-slate-950">
            <div className="container mx-auto px-6">
                <div className="mb-12">
                    <h2 className="text-3xl md:text-4xl font-heading font-bold text-eng-navy dark:text-white mb-4">Engineering Tools</h2>
                    <p className="text-slate-600 dark:text-slate-400 text-lg max-w-2xl">
                        Practical utilities to assist engineers, contractors, and home builders in material estimation.
                    </p>
                </div>

                <div className="grid lg:grid-cols-4 gap-8">
                    {/* Sidebar / Tools List */}
                    <div className="lg:col-span-1 space-y-2">
                        {TOOLS.map((tool) => (
                            <button
                                key={tool.id}
                                onClick={() => setActiveToolId(tool.id)}
                                className={`w-full text-left p-4 rounded-sm border transition-all duration-200 flex items-center justify-between group ${activeToolId === tool.id
                                    ? 'bg-eng-navy text-white border-eng-navy shadow-md'
                                    : 'bg-slate-50 dark:bg-slate-900 border-slate-200 dark:border-slate-800 hover:border-eng-blue text-slate-700 dark:text-slate-300'
                                    }`}
                            >
                                <div>
                                    <span className="font-bold block">{tool.name}</span>
                                    <span className={`text-xs ${activeToolId === tool.id ? 'text-slate-300' : 'text-slate-500'}`}>{tool.desc.substring(0, 30)}...</span>
                                </div>
                                {activeToolId === tool.id && <ChevronRight className="w-4 h-4" />}
                            </button>
                        ))}

                        <div className="p-4 bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-sm mt-8">
                            <h4 className="font-bold text-yellow-800 dark:text-yellow-500 flex items-center gap-2 mb-2 text-sm">
                                <Wrench className="w-4 h-4" /> Coming Soon
                            </h4>
                            <p className="text-xs text-yellow-700 dark:text-yellow-400">
                                More tools like Steel Weight Calculator and Footing Estimator are in development.
                            </p>
                        </div>
                    </div>

                    {/* Main Content Area */}
                    <div className="lg:col-span-3">
                        <div className="animate-fade-in">
                            <ActiveComponent />
                        </div>
                    </div>
                </div>
            </div>
        </section>
    );
};
