import React, { useState } from 'react';
import { BUILDING_DESIGNS, CONTACT_INFO } from '../constants';
import { BuildingDesign } from '../types';
import { X, Send, Phone, MapPin, Mail, CheckCircle } from 'lucide-react';

const BuildingDesigns: React.FC = () => {
    const [selectedDesign, setSelectedDesign] = useState<BuildingDesign | null>(null);
    const [showDetailsModal, setShowDetailsModal] = useState(false);
    const [showForm, setShowForm] = useState(false);
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [formData, setFormData] = useState({
        name: '',
        contact: '',
        email: '',
        address: ''
    });
    const [showFullImage, setShowFullImage] = useState(false);

    const handleSelectDesign = (design: BuildingDesign) => {
        console.log("Selected Design:", design);
        setSelectedDesign(design);
        setShowDetailsModal(true);
    };

    const handleInquire = () => {
        setShowDetailsModal(false);
        setShowForm(true);
    };

    const handleCloseDetails = () => {
        setShowDetailsModal(false);
        setSelectedDesign(null);
    };

    const handleCloseForm = () => {
        setShowForm(false);
        setSelectedDesign(null);
    };

    const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
        const { name, value } = e.target;
        setFormData(prev => ({ ...prev, [name]: value }));
    };

    const handleSendToWhatsApp = (e: React.FormEvent) => {
        e.preventDefault();
        if (!selectedDesign) return;

        const message = `*Inquiry for Building Design: ${selectedDesign.title}*\n\n` +
            `Hello Visioncraft Team,\n` +
            `I am interested in this design (ID: ${selectedDesign.id}).\n\n` +
            `*My Details:*\n` +
            `- Name: ${formData.name}\n` +
            `- Contact: ${formData.contact}\n` +
            `- Email: ${formData.email}\n` +
            `- Address: ${formData.address}\n\n` +
            `Please contact me to discuss further.`;

        const encodedMessage = encodeURIComponent(message);
        const whatsappUrl = `https://wa.me/${CONTACT_INFO.whatsapp}?text=${encodedMessage}`;

        window.open(whatsappUrl, '_blank');
        handleCloseForm();
    };

    const handleSendToEmail = async (e: React.FormEvent) => {
        e.preventDefault();
        if (!selectedDesign) return;

        setIsSubmitting(true);

        const subject = `Inquiry for Building Design: ${selectedDesign.title}`;

        try {
            const response = await fetch(`https://formsubmit.co/ajax/${CONTACT_INFO.email}`, {
                method: "POST",
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
                body: JSON.stringify({
                    _subject: subject,
                    _template: "table",
                    _captcha: "false",
                    name: formData.name,
                    contact: formData.contact,
                    email: formData.email,
                    address: formData.address,
                    design_id: selectedDesign.id,
                    design_title: selectedDesign.title
                })
            });

            if (response.ok) {
                alert("Message sent successfully! We will contact you shortly.");
                handleCloseForm();
            } else {
                alert("Failed to send message. Please try WhatsApp instead.");
            }
        } catch (error) {
            console.error(error);
            alert("An error occurred. Please check your internet connection.");
        } finally {
            setIsSubmitting(false);
        }
    };

    return (
        <section className="py-24 min-h-screen bg-slate-50 dark:bg-slate-900">
            <div className="container mx-auto px-6">
                <div className="mb-12 text-center">
                    <h2 className="text-3xl md:text-4xl font-heading font-bold text-eng-navy dark:text-slate-100 uppercase tracking-wider mb-4">
                        Curated Building Designs
                    </h2>
                    <p className="text-slate-600 dark:text-slate-400 max-w-2xl mx-auto">
                        Explore our collection of pre-approved architectural designs. Select a design you love and connect with us directly to bring it to life.
                    </p>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {BUILDING_DESIGNS.map((design) => (
                        <div key={design.id} className="group bg-white dark:bg-slate-950 rounded-xl overflow-hidden shadow-md hover:shadow-xl transition-all duration-300 flex flex-col">
                            <div className="relative h-64 overflow-hidden bg-slate-200 dark:bg-slate-800 animate-pulse">
                                <img
                                    src={design.image}
                                    alt={design.title}
                                    loading="lazy"
                                    decoding="async"
                                    onLoad={(e) => {
                                        e.currentTarget.parentElement?.classList.remove('animate-pulse');
                                        e.currentTarget.classList.remove('opacity-0');
                                    }}
                                    className="w-full h-full object-cover transition-all duration-700 group-hover:scale-105 opacity-0"
                                />
                            </div>

                            <div className="p-5 flex-1 flex flex-col">
                                <h3 className="text-lg font-bold text-eng-navy dark:text-white mb-2">{design.title}</h3>
                                <p className="text-slate-500 dark:text-slate-400 text-sm mb-4">
                                    {design.description}
                                </p>

                                <div className="flex flex-wrap gap-2 mb-4">
                                    <span className="px-3 py-1 bg-slate-100 dark:bg-slate-800 text-slate-700 dark:text-slate-300 text-xs font-medium rounded-full">
                                        {design.category}
                                    </span>
                                </div>

                                <button
                                    onClick={() => handleSelectDesign(design)}
                                    className="mt-auto w-full py-3 bg-eng-blue hover:bg-eng-navy text-white text-sm font-bold uppercase tracking-widest transition-colors flex items-center justify-center gap-2 rounded-lg"
                                >
                                    Select This Design <CheckCircle className="w-4 h-4" />
                                </button>
                            </div>
                        </div>
                    ))}
                </div>
            </div>

            {/* Full Image Modal */}
            {showFullImage && selectedDesign && (
                <div
                    className="fixed inset-0 z-[60] bg-black/90 flex items-center justify-center p-4 backdrop-blur-sm animate-fade-in"
                    onClick={() => setShowFullImage(false)}
                >
                    <button
                        onClick={() => setShowFullImage(false)}
                        className="absolute top-6 right-6 p-2 text-white/70 hover:text-white bg-black/50 rounded-full hover:bg-black/70 transition-all"
                    >
                        <X className="w-8 h-8" />
                    </button>
                    <img
                        src={selectedDesign.image}
                        alt={selectedDesign.title}
                        loading="lazy"
                        decoding="async"
                        className="max-w-full max-h-screen object-contain rounded-lg shadow-2xl animate-fade-in"
                    />
                </div>
            )}

            {/* Detailed Info Modal */}
            {showDetailsModal && selectedDesign && (
                <div className="fixed inset-0 z-50 overflow-y-auto bg-black/60 backdrop-blur-sm">
                    <div className="min-h-full flex items-center justify-center p-4">
                        <div className="bg-white dark:bg-slate-900 rounded-2xl w-full max-w-6xl relative animate-fade-in-up my-8 shadow-2xl border border-slate-200 dark:border-slate-700">
                            <button
                                onClick={() => setShowDetailsModal(false)}
                                className="absolute top-4 right-4 p-2 text-slate-400 hover:text-slate-600 dark:hover:text-slate-300 transition-colors bg-slate-100 dark:bg-slate-800 rounded-full hover:rotate-90 duration-300 z-10"
                            >
                                <X className="w-6 h-6" />
                            </button>

                            <div className="grid md:grid-cols-2 gap-6 p-8">
                                {/* Left: Image */}
                                <div
                                    className="relative w-full md:h-full rounded-lg overflow-hidden bg-slate-200 dark:bg-slate-800 group cursor-zoom-in animate-pulse"
                                    onClick={() => setShowFullImage(true)}
                                >
                                    <div className="absolute inset-0 bg-black/0 group-hover:bg-black/10 transition-colors duration-300 z-10 flex items-center justify-center">
                                        <span className="opacity-0 group-hover:opacity-100 bg-black/50 text-white px-4 py-2 rounded-full text-sm font-medium transition-all transform scale-90 group-hover:scale-100 pointer-events-none">
                                            View Full Image
                                        </span>
                                    </div>
                                    <img
                                        src={selectedDesign.image}
                                        alt={selectedDesign.title}
                                        loading="lazy"
                                        decoding="async"
                                        onLoad={(e) => {
                                            e.currentTarget.parentElement?.classList.remove('animate-pulse');
                                            e.currentTarget.classList.remove('opacity-0');
                                        }}
                                        className="w-full h-auto md:h-full md:object-cover transition-all duration-500 group-hover:scale-105 opacity-0"
                                    />
                                </div>

                                {/* Right: Details */}
                                <div className="flex flex-col h-full overflow-hidden">
                                    <div className="flex-1 max-h-[60vh] md:max-h-[65vh] overflow-y-auto pr-4 custom-scrollbar">
                                        <h2 className="text-3xl font-bold text-eng-navy dark:text-white mb-4">{selectedDesign.title}</h2>

                                        {/* Specification Cards */}
                                        <div className="grid grid-cols-2 gap-3 mb-6">
                                            {selectedDesign.constructionTime && (
                                                <div className="bg-slate-50 dark:bg-slate-800 p-3 rounded-lg text-center border border-slate-100 dark:border-slate-700">
                                                    <div className="text-xs text-slate-500 dark:text-slate-400 mb-1 uppercase tracking-wide">Time to build</div>
                                                    <div className="text-sm font-bold text-eng-navy dark:text-white">{selectedDesign.constructionTime}</div>
                                                </div>
                                            )}
                                            {selectedDesign.area && (
                                                <div className="bg-slate-50 dark:bg-slate-800 p-3 rounded-lg text-center border border-slate-100 dark:border-slate-700">
                                                    <div className="text-xs text-slate-500 dark:text-slate-400 mb-1 uppercase tracking-wide">Area (Sq. ft)</div>
                                                    <div className="text-sm font-bold text-eng-navy dark:text-white">{selectedDesign.area}</div>
                                                </div>
                                            )}
                                            {selectedDesign.houseStyle && (
                                                <div className="bg-slate-50 dark:bg-slate-800 p-3 rounded-lg text-center border border-slate-100 dark:border-slate-700">
                                                    <div className="text-xs text-slate-500 dark:text-slate-400 mb-1 uppercase tracking-wide">House Style</div>
                                                    <div className="text-sm font-bold text-eng-navy dark:text-white">{selectedDesign.houseStyle}</div>
                                                </div>
                                            )}
                                            {selectedDesign.stories && (
                                                <div className="bg-slate-50 dark:bg-slate-800 p-3 rounded-lg text-center border border-slate-100 dark:border-slate-700">
                                                    <div className="text-xs text-slate-500 dark:text-slate-400 mb-1 uppercase tracking-wide">Stories</div>
                                                    <div className="text-sm font-bold text-eng-navy dark:text-white">{selectedDesign.stories}</div>
                                                </div>
                                            )}
                                            {selectedDesign.facingDirection && (
                                                <div className="bg-slate-50 dark:bg-slate-800 p-3 rounded-lg text-center border border-slate-100 dark:border-slate-700">
                                                    <div className="text-xs text-slate-500 dark:text-slate-400 mb-1 uppercase tracking-wide">Facing Direction</div>
                                                    <div className="text-sm font-bold text-eng-navy dark:text-white">{selectedDesign.facingDirection}</div>
                                                </div>
                                            )}
                                            {selectedDesign.plinthArea && (
                                                <div className="bg-slate-50 dark:bg-slate-800 p-3 rounded-lg text-center border border-slate-100 dark:border-slate-700">
                                                    <div className="text-xs text-slate-500 dark:text-slate-400 mb-1 uppercase tracking-wide">Plinth Area</div>
                                                    <div className="text-sm font-bold text-eng-navy dark:text-white">{selectedDesign.plinthArea}</div>
                                                </div>
                                            )}
                                        </div>

                                        {/* Description */}
                                        <div className="mb-6">
                                            <h3 className="text-sm font-bold text-eng-navy dark:text-white mb-2 uppercase tracking-wide">Description</h3>
                                            <p className="text-slate-600 dark:text-slate-300 text-sm leading-relaxed mb-4 whitespace-pre-line">
                                                {selectedDesign.fullDescription || selectedDesign.description}
                                            </p>

                                            {selectedDesign.roomSpecifications && selectedDesign.roomSpecifications.length > 0 && (
                                                <div className="mt-6 bg-slate-50 dark:bg-slate-800/50 p-4 rounded-lg">
                                                    <h3 className="text-sm font-bold text-eng-navy dark:text-white mb-3 uppercase tracking-wide">Room Specifications</h3>
                                                    <div className="grid gap-2">
                                                        {selectedDesign.roomSpecifications.map((room, idx) => (
                                                            <div key={idx} className="text-xs md:text-sm text-slate-600 dark:text-slate-400 border-b border-slate-200 dark:border-slate-700 last:border-0 pb-2 last:pb-0">
                                                                {room}
                                                            </div>
                                                        ))}
                                                    </div>
                                                </div>
                                            )}
                                        </div>
                                    </div>

                                    {/* Action Button - Stationary */}
                                    <div className="mt-6 pt-4 border-t border-slate-100 dark:border-slate-800">
                                        <button
                                            onClick={handleInquire}
                                            className="w-full py-4 bg-eng-blue hover:bg-eng-navy text-white font-bold uppercase tracking-widest transition-all flex items-center justify-center gap-2 rounded-lg shadow-lg hover:shadow-xl duration-300"
                                        >
                                            Inquire About This Design <CheckCircle className="w-5 h-5" />
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            )}

            {/* Inquiry Modal */}
            {showForm && selectedDesign && (
                <div className="fixed inset-0 z-50 overflow-y-auto bg-black/60 backdrop-blur-sm">
                    <div className="min-h-full flex items-center justify-center p-4">
                        <div className="bg-white dark:bg-slate-900 w-full max-w-lg rounded-sm shadow-2xl border border-slate-200 dark:border-slate-700 relative animate-fade-in-up">
                            <button
                                onClick={handleCloseForm}
                                className="absolute top-4 right-4 text-slate-400 hover:text-red-500 transition-colors bg-white dark:bg-slate-800 rounded-full p-1"
                            >
                                <X className="w-6 h-6" />
                            </button>

                            <div className="p-8">
                                <h3 className="text-2xl font-bold text-eng-navy dark:text-white mb-2">Inquire About Design</h3>
                                <p className="text-eng-blue font-medium mb-6 uppercase tracking-wide text-sm">{selectedDesign.title}</p>

                                <form onSubmit={handleSendToWhatsApp} className="space-y-4">
                                    <div>
                                        <label className="block text-xs font-bold text-slate-500 uppercase tracking-wider mb-1">Full Name</label>
                                        <input
                                            name="name"
                                            type="text"
                                            required
                                            value={formData.name}
                                            onChange={handleInputChange}
                                            className="w-full px-4 py-3 bg-slate-50 dark:bg-slate-800 border border-slate-300 dark:border-slate-700 rounded-sm focus:border-eng-blue focus:outline-none transition-colors"
                                            placeholder="Enter your name"
                                        />
                                    </div>

                                    <div className="grid grid-cols-2 gap-4">
                                        <div>
                                            <label className="block text-xs font-bold text-slate-500 uppercase tracking-wider mb-1">Contact Number</label>
                                            <input
                                                name="contact"
                                                type="tel"
                                                required
                                                value={formData.contact}
                                                onChange={handleInputChange}
                                                className="w-full px-4 py-3 bg-slate-50 dark:bg-slate-800 border border-slate-300 dark:border-slate-700 rounded-sm focus:border-eng-blue focus:outline-none transition-colors"
                                                placeholder="Mobile Number"
                                            />
                                        </div>
                                        <div>
                                            <label className="block text-xs font-bold text-slate-500 uppercase tracking-wider mb-1">Email (Optional)</label>
                                            <input
                                                name="email"
                                                type="email"
                                                value={formData.email}
                                                onChange={handleInputChange}
                                                className="w-full px-4 py-3 bg-slate-50 dark:bg-slate-800 border border-slate-300 dark:border-slate-700 rounded-sm focus:border-eng-blue focus:outline-none transition-colors"
                                                placeholder="Email Address"
                                            />
                                        </div>
                                    </div>

                                    <div>
                                        <label className="block text-xs font-bold text-slate-500 uppercase tracking-wider mb-1">Project Site Address</label>
                                        <input
                                            name="address"
                                            type="text"
                                            required
                                            value={formData.address}
                                            onChange={handleInputChange}
                                            className="w-full px-4 py-3 bg-slate-50 dark:bg-slate-800 border border-slate-300 dark:border-slate-700 rounded-sm focus:border-eng-blue focus:outline-none transition-colors"
                                            placeholder="Where will this be built?"
                                        />
                                    </div>

                                    <div className="pt-4">
                                        <div className="pt-4 grid grid-cols-2 gap-4">
                                            <button
                                                type="submit"
                                                className="col-span-2 md:col-span-1 py-4 bg-[#25D366] hover:bg-[#128C7E] text-white font-bold uppercase tracking-widest transition-colors flex items-center justify-center gap-2 shadow-lg hover:shadow-xl translate-y-0 hover:-translate-y-1 transform duration-300"
                                            >
                                                <Send className="w-5 h-5" /> WhatsApp
                                            </button>
                                            <button
                                                type="button"
                                                onClick={handleSendToEmail}
                                                disabled={isSubmitting}
                                                className={`col-span-2 md:col-span-1 py-4 bg-eng-navy hover:bg-slate-800 text-white font-bold uppercase tracking-widest transition-colors flex items-center justify-center gap-2 shadow-lg hover:shadow-xl translate-y-0 hover:-translate-y-1 transform duration-300 ${isSubmitting ? 'opacity-75 cursor-not-allowed' : ''}`}
                                            >
                                                {isSubmitting ? (
                                                    <div className="w-5 h-5 border-2 border-white/30 border-t-white rounded-full animate-spin" />
                                                ) : (
                                                    <>
                                                        <Mail className="w-5 h-5" /> Email
                                                    </>
                                                )}
                                            </button>
                                            <p className="col-span-2 text-center text-xs text-slate-400 mt-1">
                                                Choose your preferred method to send the inquiry.
                                            </p>
                                        </div>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
            )}
        </section>
    );
};

export default BuildingDesigns;
