import React, { useState, useEffect } from 'react';
import { HashRouter, Routes, Route, Link, useLocation } from 'react-router-dom';
import { Menu, X, Moon, Sun, ChevronRight, MapPin, Phone, Mail, FileCheck } from 'lucide-react';
import { BlueprintBackground } from './components/BlueprintBackground';
import { ToolsPage } from './components/ToolsPage';
import BuildingDesigns from './components/BuildingDesigns';
import { COMPANY_NAME, TAGLINE, SERVICES, TEAM, PUBLICATIONS, CONTACT_INFO } from './constants';

// --- Shared Components ---

const SectionHeading: React.FC<{ title: string; subtitle?: string; align?: 'left' | 'center' }> = ({ title, subtitle, align = 'center' }) => (
  <div className={`mb-12 ${align === 'center' ? 'text-center' : 'text-left'}`}>
    <h2 className="text-2xl md:text-3xl font-heading font-bold text-eng-navy dark:text-slate-100 uppercase tracking-wider relative inline-block">
      {title}
      <span className="absolute -bottom-2 left-0 w-full h-1 bg-eng-blue opacity-50"></span>
    </h2>
    {subtitle && <p className="mt-4 text-slate-600 dark:text-slate-400 max-w-2xl mx-auto">{subtitle}</p>}
  </div>
);

const Button: React.FC<{ children: React.ReactNode; variant?: 'primary' | 'outline' | 'outline-white'; onClick?: () => void; to?: string }> = ({ children, variant = 'primary', onClick, to }) => {
  const baseClasses = "inline-flex items-center justify-center px-8 py-4 text-xs font-bold tracking-widest uppercase transition-all duration-300";
  const variants = {
    primary: "bg-eng-blue hover:bg-eng-navy text-white",
    outline: "border border-eng-navy text-eng-navy hover:bg-eng-navy hover:text-white dark:border-white dark:text-white dark:hover:bg-white dark:hover:text-eng-navy",
    "outline-white": "border border-white text-white hover:bg-white hover:text-eng-navy"
  };

  if (to) {
    return <Link to={to} className={`${baseClasses} ${variants[variant]}`}>{children}</Link>;
  }
  return <button onClick={onClick} className={`${baseClasses} ${variants[variant]}`}>{children}</button>;
};

// --- Page Components ---

const VIDEO_PLAYLIST = ['/assets/hero.mp4', '/assets/hero2.mp4'];

const Home: React.FC = () => {
  const [currentVideoIndex, setCurrentVideoIndex] = useState(0);
  const [isVideoLoaded, setIsVideoLoaded] = useState(false);
  const videoRef = React.useRef<HTMLVideoElement>(null);
  const hasStartedRef = React.useRef(false);

  useEffect(() => {
    // Play immediately on mount
    if (videoRef.current) {
      videoRef.current.play().catch(e => console.log("Playback error:", e));
      hasStartedRef.current = true;
    }
  }, []);

  const handleVideoEnd = () => {
    setIsVideoLoaded(false); // Reset loading state for the next video
    const nextIndex = (currentVideoIndex + 1) % VIDEO_PLAYLIST.length;
    setCurrentVideoIndex(nextIndex);
  };

  useEffect(() => {
    // Play immediately for all subsequent video changes
    if (hasStartedRef.current && videoRef.current) {
      videoRef.current.muted = true; // Force mute
      videoRef.current.play().catch(e => console.log("Playback error:", e));
    }
  }, [currentVideoIndex]);

  return (
    <>
      {/* Cinematic Hero - Arup Style with User Video */}
      <section className="relative h-screen flex items-center justify-center overflow-hidden bg-slate-900">
        <div className="absolute inset-0 z-0 bg-slate-900">
          {/* Show a placeholder/poster or solid color until video is loaded */}
          <video
            ref={videoRef}
            src={VIDEO_PLAYLIST[currentVideoIndex]}
            onEnded={handleVideoEnd}
            onCanPlay={() => setIsVideoLoaded(true)}
            autoPlay
            muted
            loop={false}
            playsInline
            preload="auto"
            poster="/assets/posterimage.webp"
            className={`w-full h-full object-cover transition-opacity duration-1000 ${isVideoLoaded ? 'opacity-60' : 'opacity-0'}`}
          >
            {/* Fallback source for initial load redundancy */}
            <source src={VIDEO_PLAYLIST[currentVideoIndex]} type="video/mp4" />
          </video>
          <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-black/40 to-black/10"></div>
        </div>

        <div className="container mx-auto px-6 relative z-10 text-center md:text-left">
          <h1 className="text-3xl md:text-4xl lg:text-5xl font-heading font-bold text-white leading-tight mb-4 tracking-tight">
            SHAPING A<br />
            BETTER WORLD
          </h1>
          <p className="text-lg md:text-xl text-slate-200 mb-8 max-w-2xl font-light leading-relaxed">
            We combine engineering excellence with sustainable innovation to deliver resilient infrastructure for tomorrow.
          </p>
          <div className="flex flex-col sm:flex-row gap-6">
            <Button variant="outline-white" to="/contact">Work With Us</Button>
          </div>
        </div>
      </section>

      {/* Minimalist Services - Borderless Grid */}
      <section className="py-24 bg-white dark:bg-slate-950">
        <div className="container mx-auto px-6">
          <SectionHeading title="Expertise" subtitle="Comprehensive solutions for the built environment." align="left" />
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-12 border-t border-slate-200 dark:border-slate-800 pt-12">
            {SERVICES.map((s) => (
              <div key={s.id} className="group cursor-pointer">
                <s.icon className="w-12 h-12 text-eng-navy dark:text-white mb-6 group-hover:text-eng-blue transition-colors" />
                <h3 className="text-2xl font-bold text-eng-navy dark:text-white mb-4 group-hover:translate-x-2 transition-transform">{s.title}</h3>
                <p className="text-slate-600 dark:text-slate-400 leading-relaxed mb-4">{s.description}</p>
                <Link to="/services" className="inline-flex items-center text-sm font-semibold text-eng-blue uppercase tracking-widest group-hover:underline">
                  Explore <ChevronRight className="w-4 h-4 ml-1" />
                </Link>
              </div>
            ))}
          </div>
        </div>
      </section>


    </>
  );
};

const About: React.FC = () => (
  <section className="py-20 min-h-screen bg-slate-50 dark:bg-slate-950">
    <div className="container mx-auto px-6">
      <SectionHeading title="About Us" subtitle="Innovation meets integrity in modern civil engineering." />
      <div className="grid md:grid-cols-2 gap-12 items-center">
        <div>
          <h3 className="text-2xl font-heading font-bold mb-4 text-eng-navy dark:text-white uppercase tracking-wide">Engineering the Future with Precision</h3>
          <p className="text-slate-600 dark:text-slate-300 mb-6 leading-relaxed text-lg">
            Founded on the principles of modern innovation and structural integrity, {COMPANY_NAME} is a new-age consultancy dedicated to solving today’s complex construction challenges. We believe that great design is a balance of safety, sustainability, and cost-efficiency.
          </p>
          <p className="text-slate-600 dark:text-slate-300 mb-8 leading-relaxed text-lg border-l-4 border-eng-blue pl-6 italic">
            Our team brings together high-level academic expertise and a commitment to rigorous standards, including National and International codes. We don't just design structures; we build the foundations for a safer tomorrow.
          </p>
        </div>
        <div className="relative">
          <div className="absolute inset-0 bg-eng-navy transform translate-x-4 translate-y-4 -z-10"></div>
          <img src="/assets/aboutus.png" alt="About us" className="w-full grayscale hover:grayscale-0 transition-all duration-500" />
        </div>
      </div>
    </div>
  </section>
);

const ServicesPage: React.FC = () => (
  <section className="py-24 min-h-screen bg-white dark:bg-slate-950">
    <div className="container mx-auto px-6">
      <SectionHeading title="Our Services" subtitle="Specialized engineering solutions utilizing state-of-the-art software and methodologies." />
      <div className="grid gap-16 mt-16">
        {SERVICES.map((service, index) => (
          <div key={service.id} className={`flex flex-col md:flex-row gap-12 items-start ${index % 2 === 1 ? 'md:flex-row-reverse' : ''}`}>
            <div className="w-full md:w-1/3 bg-slate-100 dark:bg-slate-900 aspect-video flex items-center justify-center rounded-sm overflow-hidden relative">
              {service.image ? (
                <img src={service.image} alt={service.title} className="w-full h-full object-cover hover:scale-105 transition-transform duration-700" />
              ) : (
                <service.icon className="w-24 h-24 text-eng-navy/20 dark:text-white/20" />
              )}
            </div>
            <div className="flex-1">
              <h3 className="text-3xl font-bold text-eng-navy dark:text-white mb-4">{service.title}</h3>
              <p className="text-slate-600 dark:text-slate-300 mb-8 text-lg leading-relaxed">{service.description}</p>
              <ul className="grid sm:grid-cols-2 gap-4">
                {service.details.map((detail, idx) => (
                  <li key={idx} className="flex items-center text-slate-800 dark:text-slate-200 font-medium">
                    <div className="w-2 h-2 bg-eng-blue rounded-full mr-4"></div>
                    {detail}
                  </li>
                ))}
              </ul>
            </div>
          </div>
        ))}
      </div>
    </div>
  </section>
);


const TeamPage: React.FC = () => (
  <section className="py-20 min-h-screen bg-white dark:bg-slate-950">
    <div className="container mx-auto px-6">
      <SectionHeading title="Our Team" subtitle="Qualified experts dedicated to the highest standards of engineering." />
      <div className="grid md:grid-cols-3 gap-8 max-w-6xl mx-auto">
        {TEAM.map((member) => (
          <div key={member.id} className="text-center group p-6 border border-slate-100 dark:border-slate-800 bg-slate-50 dark:bg-slate-900 transition-all hover:shadow-lg">
            <div className="w-32 h-32 mx-auto mb-4 overflow-hidden rounded-full border-4 border-white dark:border-slate-800 shadow-sm">
              <img src={member.image} alt={member.name} className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500" />
            </div>
            <h3 className="text-xl font-bold text-eng-navy dark:text-white">{member.name}</h3>
            <p className="text-eng-blue font-medium mb-2">{member.designation}</p>
            <div className="text-sm text-slate-600 dark:text-slate-400 font-mono space-y-1">
              <p>{member.qualification}</p>
              <p>{member.necReg}</p>
            </div>
          </div>
        ))}
      </div>

      <div className="mt-20">
        <SectionHeading title="Research & Publications" subtitle="Contributing to the body of engineering knowledge." />
        <div className="max-w-4xl mx-auto bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 divide-y divide-slate-200 dark:divide-slate-800">
          {PUBLICATIONS.map((pub) => (
            <div key={pub.id} className="p-6 flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4 hover:bg-white dark:hover:bg-slate-800 transition-colors">
              <div>
                <h4 className="font-semibold text-eng-navy dark:text-slate-200">{pub.title}</h4>
                <p className="text-sm text-slate-500">{pub.journal} • {pub.year}</p>
              </div>
              <a href={pub.link} className="text-xs font-bold text-eng-blue uppercase tracking-wide border border-eng-blue px-3 py-1 hover:bg-eng-blue hover:text-white transition-colors">
                Read Paper
              </a>
            </div>
          ))}
        </div>
      </div>
    </div>
  </section>
);

const Contact: React.FC = () => {
  const [formData, setFormData] = useState({
    name: '',
    phone: '',
    email: '',
    message: ''
  });
  const [isSubmitting, setIsSubmitting] = useState(false);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsSubmitting(true);

    try {
      const response = await fetch(`https://formsubmit.co/ajax/${CONTACT_INFO.email}`, {
        method: "POST",
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        },
        body: JSON.stringify({
          _subject: `New Inquiry from ${formData.name}`,
          _template: "table",
          _captcha: "false",
          name: formData.name,
          phone: formData.phone,
          email: formData.email,
          message: formData.message
        })
      });

      if (response.ok) {
        alert("Thank you. Your message has been sent successfully. We will contact you shortly.");
        setFormData({ name: '', phone: '', email: '', message: '' });
      } else {
        alert("Failed to send message. Please try again later or contact us directly via phone.");
      }
    } catch (error) {
      console.error(error);
      alert("An error occurred. Please check your internet connection.");
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <section className="py-20 min-h-screen bg-slate-50 dark:bg-slate-950">
      <div className="container mx-auto px-6">
        <SectionHeading title="Contact Us" subtitle="Get in touch for professional consultation and inquiries." />
        <div className="grid lg:grid-cols-2 gap-12 max-w-6xl mx-auto">
          <div className="bg-white dark:bg-slate-900 p-8 shadow-sm border border-slate-200 dark:border-slate-800">
            <h3 className="text-xl font-bold text-eng-navy dark:text-white mb-6">Send an Inquiry</h3>
            <form className="space-y-6" onSubmit={handleSubmit}>
              <div className="grid md:grid-cols-2 gap-6">
                <div>
                  <label className="block text-sm font-semibold text-slate-700 dark:text-slate-300 mb-2">Name</label>
                  <input
                    type="text"
                    name="name"
                    required
                    value={formData.name}
                    onChange={handleInputChange}
                    className="w-full px-4 py-3 bg-slate-50 dark:bg-slate-800 border border-slate-300 dark:border-slate-700 rounded-sm focus:outline-none focus:border-eng-navy transition-colors"
                  />
                </div>
                <div>
                  <label className="block text-sm font-semibold text-slate-700 dark:text-slate-300 mb-2">Phone</label>
                  <input
                    type="tel"
                    name="phone"
                    required
                    value={formData.phone}
                    onChange={handleInputChange}
                    className="w-full px-4 py-3 bg-slate-50 dark:bg-slate-800 border border-slate-300 dark:border-slate-700 rounded-sm focus:outline-none focus:border-eng-navy transition-colors"
                  />
                </div>
              </div>
              <div>
                <label className="block text-sm font-semibold text-slate-700 dark:text-slate-300 mb-2">Email</label>
                <input
                  type="email"
                  name="email"
                  required
                  value={formData.email}
                  onChange={handleInputChange}
                  className="w-full px-4 py-3 bg-slate-50 dark:bg-slate-800 border border-slate-300 dark:border-slate-700 rounded-sm focus:outline-none focus:border-eng-navy transition-colors"
                />
              </div>
              <div>
                <label className="block text-sm font-semibold text-slate-700 dark:text-slate-300 mb-2">Message</label>
                <textarea
                  name="message"
                  required
                  value={formData.message}
                  onChange={handleInputChange}
                  rows={4}
                  className="w-full px-4 py-3 bg-slate-50 dark:bg-slate-800 border border-slate-300 dark:border-slate-700 rounded-sm focus:outline-none focus:border-eng-navy transition-colors"
                ></textarea>
              </div>
              <button
                type="submit"
                disabled={isSubmitting}
                className={`inline-flex items-center justify-center px-8 py-4 text-xs font-bold tracking-widest uppercase transition-all duration-300 bg-eng-blue hover:bg-eng-navy text-white ${isSubmitting ? 'opacity-70 cursor-not-allowed' : ''}`}
              >
                {isSubmitting ? 'Sending...' : 'Submit Inquiry'}
              </button>
            </form>
          </div>

          <div className="space-y-8">
            <div className="bg-white dark:bg-slate-900 p-8 shadow-sm border border-slate-200 dark:border-slate-800">
              <h3 className="text-xl font-bold text-eng-navy dark:text-white mb-6">Office Information</h3>
              <div className="space-y-4 text-slate-600 dark:text-slate-300">
                <div className="flex items-start gap-4">
                  <MapPin className="w-5 h-5 text-eng-blue mt-1 shrink-0" />
                  <p>{CONTACT_INFO.address}</p>
                </div>
                <div className="flex items-center gap-4">
                  <Phone className="w-5 h-5 text-eng-blue shrink-0" />
                  <p>+{CONTACT_INFO.whatsapp}</p>
                </div>
                <div className="flex items-center gap-4">
                  <Mail className="w-5 h-5 text-eng-blue shrink-0" />
                  <p>{CONTACT_INFO.email}</p>
                </div>
              </div>
              <div className="mt-8 pt-8 border-t border-slate-100 dark:border-slate-800">
                <p className="text-sm font-mono text-slate-500">
                  Consultation available by appointment only.<br />
                  Sunday - Friday: 10:00 AM - 5:00 PM
                </p>
              </div>
            </div>

            {/* Google Map */}
            <div className="h-64 w-full overflow-hidden border border-slate-200 dark:border-slate-800 shadow-sm">
              <iframe
                src="https://www.google.com/maps/embed?pb=!1m17!1m12!1m3!1d1766.292203213877!2d85.3406144711252!3d27.69923699789525!2m3!1f0!2f0!3f0!3m2!1i1024!2i768!4f13.1!3m2!1m1!2zMjfCsDQxJzU3LjMiTiA4NcKwMjAnMjkuMCJF!5e0!3m2!1sen!2sus!4v1771227257748!5m2!1sen!2sus"
                width="100%"
                height="100%"
                style={{ border: 0 }}
                allowFullScreen
                loading="lazy"
                referrerPolicy="no-referrer-when-downgrade"
                title="Visioncraft Civil Solutions Office - Infrastructure Tower, New Baneshwor"
              ></iframe>
            </div>
          </div>
        </div>
      </div>
    </section>
  );
};

const Navbar: React.FC<{ darkMode: boolean; toggleTheme: () => void }> = ({ darkMode, toggleTheme }) => {
  const [isOpen, setIsOpen] = useState(false);
  const [scrolled, setScrolled] = useState(false);
  const location = useLocation();
  const isHome = location.pathname === '/';

  useEffect(() => {
    const handleScroll = () => setScrolled(window.scrollY > 50);
    window.addEventListener('scroll', handleScroll);
    return () => window.removeEventListener('scroll', handleScroll);
  }, []);

  const navLinks = [
    { name: 'Home', path: '/' },
    { name: 'About', path: '/about' },
    { name: 'Services', path: '/services' },
    { name: 'House Plans', path: '/designs' },
    { name: 'Tools', path: '/tools' },
    { name: 'Team', path: '/team' },
    { name: 'Contact', path: '/contact' },
  ];

  const headerBg = scrolled || !isHome ? 'bg-white/95 dark:bg-slate-950/95 backdrop-blur-md shadow-sm' : 'bg-transparent';
  const textColor = scrolled || !isHome ? 'text-slate-800 dark:text-slate-100' : 'text-white';
  const logoColor = scrolled || !isHome ? 'text-eng-navy dark:text-white' : 'text-white';

  return (
    <header className={`fixed top-0 left-0 w-full z-50 transition-all duration-300 ${headerBg}`}>
      <div className="container mx-auto px-6 py-4 flex items-center justify-between">
        <Link to="/" className="flex items-center gap-3">
          {/* Auto-switching logo or text based on background */}
          <div className={`font-heading font-bold text-2xl tracking-tighter uppercase ${logoColor}`}>
            Visioncraft<span className="text-eng-blue">.</span>
          </div>
        </Link>

        {/* Desktop Nav */}
        <nav className="hidden md:flex items-center gap-8">
          {navLinks.map((link) => (
            <Link
              key={link.name}
              to={link.path}
              className={`text-sm font-medium tracking-widest uppercase hover:text-eng-blue transition-colors ${textColor}`}
            >
              {link.name}
            </Link>
          ))}
          <button onClick={toggleTheme} className={`p-2 rounded-full hover:bg-white/10 ${textColor}`}>
            {darkMode ? <Sun className="w-5 h-5" /> : <Moon className="w-5 h-5" />}
          </button>
        </nav>

        {/* Mobile Toggle */}
        <button className={`md:hidden ${textColor}`} onClick={() => setIsOpen(!isOpen)}>
          {isOpen ? <X /> : <Menu />}
        </button>
      </div>

      {/* Mobile Nav Overlay */}
      {isOpen && (
        <div className="absolute top-full left-0 w-full bg-white dark:bg-slate-950 border-t border-slate-100 dark:border-slate-800 shadow-xl py-6 px-6 flex flex-col gap-4 md:hidden">
          {navLinks.map(link => (
            <Link key={link.name} to={link.path} onClick={() => setIsOpen(false)} className="text-lg font-bold text-eng-navy dark:text-white uppercase tracking-wider">
              {link.name}
            </Link>
          ))}
        </div>
      )}
    </header>
  );
};

const Footer: React.FC = () => (
  <footer className="bg-eng-navy text-slate-300 py-12 border-t border-slate-800">
    <div className="container mx-auto px-6 grid md:grid-cols-4 gap-8 text-sm">
      <div className="col-span-1 md:col-span-2">
        <h2 className="text-white font-heading font-bold text-lg mb-4">{COMPANY_NAME}</h2>
        <p className="max-w-md mb-6 text-slate-400">
          {TAGLINE}. Professional civil engineering services specializing in structural design, geotechnical investigation, and project management.
        </p>
        <div className="flex gap-4">
          {/* Social Media Links */}
          <a
            href={CONTACT_INFO.facebook}
            target="_blank"
            rel="noopener noreferrer"
            className="w-8 h-8 bg-slate-800 flex items-center justify-center hover:bg-eng-blue transition-colors cursor-pointer text-xs font-bold"
            aria-label="Facebook"
          >
            fb
          </a>
        </div>
      </div>

      <div>
        <h3 className="text-white font-bold mb-4 uppercase tracking-wider">Quick Links</h3>
        <ul className="space-y-2">
          <li><Link to="/about" className="hover:text-white transition-colors">About Us</Link></li>
          <li><Link to="/services" className="hover:text-white transition-colors">Services</Link></li>
          <li><Link to="/team" className="hover:text-white transition-colors">Team</Link></li>
        </ul>
      </div>

      <div>
        <h3 className="text-white font-bold mb-4 uppercase tracking-wider">Legal</h3>
        <ul className="space-y-2">
          <li><span className="hover:text-white transition-colors cursor-pointer">Privacy Policy</span></li>
          <li><span className="hover:text-white transition-colors cursor-pointer">Terms of Service</span></li>
          <li className="pt-4 text-slate-500">© {new Date().getFullYear()} Visioncraft. All Rights Reserved.</li>
        </ul>
      </div>
    </div>
  </footer>
);

const ScrollToTop = () => {
  const { pathname } = useLocation();
  useEffect(() => {
    window.scrollTo(0, 0);
  }, [pathname]);
  return null;
};

// --- Main App Component ---

const App: React.FC = () => {
  const [darkMode, setDarkMode] = useState(false);

  useEffect(() => {
    if (darkMode) {
      document.documentElement.classList.add('dark');
    } else {
      document.documentElement.classList.remove('dark');
    }
  }, [darkMode]);

  return (
    <HashRouter>
      <ScrollToTop />
      <div className="min-h-screen flex flex-col font-sans bg-white dark:bg-slate-950 text-slate-900 dark:text-slate-100 transition-colors duration-300">
        <Navbar darkMode={darkMode} toggleTheme={() => setDarkMode(!darkMode)} />
        <main className="flex-grow">
          <Routes>
            <Route path="/" element={<Home />} />
            <Route path="/about" element={<About />} />
            <Route path="/services" element={<ServicesPage />} />
            <Route path="/designs" element={<BuildingDesigns />} />
            <Route path="/tools" element={<ToolsPage />} />
            <Route path="/team" element={<TeamPage />} />
            <Route path="/contact" element={<Contact />} />
          </Routes>
        </main>
        <Footer />
      </div>
    </HashRouter>
  );
};

export default App;